
// ****************************************************************************
//                                 
//                         Real numbers (floating-point)
//
// ****************************************************************************

/* Configuration:

#define BASE_TYPE	1		// type of base integer number (1 = byte u8, 2 = word u16, 4 = dword u32)

// real number format
//   xxx bits mantissa (with hidden bit "1")
//   yyy bits exponent
//   1 bits sign
#define R_BYTES		11		// size of number in bytes
#define R_EXP_BITS	16		// number of bits per exponent (2..32 bits)
//#define R_COMP_BIAS		// flag - use compatible exponent bias (0x7fff, or more optimal 0x8000 otherwise)

// memory number format (must not be larger than internal format)
#define M_BYTES		8		// size of number in bytes
#define M_EXP_BITS	11		// number of bits per exponent (1..32 bits)
#define M_COMP_BIAS			// flag - uncomment this to use compatible exponent bias (0x7fff, or more optimal 0x8000 otherwise)

*/

// angle unit
#define UNIT_DEG	0	// degrees
#define UNIT_RAD	1	// radians
#define UNIT_GRAD	2	// grads

// current angle unit (UNIT_*)
extern u8 Unit;

// ===========================================================================
//                        Real number configuration
// ===========================================================================

// type of base integer
#if BASE_TYPE == 1
typedef u8 r_base;			// type of base integer unsigned
typedef s8 r_bases;			// type of base integer signed
typedef u16 r_dbl;			// type of double integer unsigned

#elif BASE_TYPE == 2
typedef u16 r_base;			// type of base integer unsigned
typedef s16 r_bases;			// type of base integer signed
typedef u32 r_dbl;			// type of double integer unsigned

#elif BASE_TYPE == 4
typedef u32 r_base;			// type of base integer unsigned
typedef s32 r_bases;			// type of base integer signed
typedef u64 r_dbl;			// type of double integer unsigned

#else
#error R_TYPE not valid
#endif

// exponent type
#if R_EXP_BITS <= 8
typedef u8 r_exp;
typedef s8 r_exps;
typedef s16 r_expd;
#define R_EXP_MAXBITS 8

#elif R_EXP_BITS <= 16
typedef u16 r_exp;
typedef s16 r_exps;
typedef s32 r_expd;
#define R_EXP_MAXBITS 16

#elif R_EXP_BITS <= 32
typedef u32 r_exp;
typedef s32 r_exps;
typedef s32 r_expd;
#define R_EXP_MAXBITS 32

#else
#error R_EXP_BITS is > 32!
#endif

// base segment
#define R_BASE_BYTES 	BASE_TYPE		// size of base segment in number of bytes
#define R_BASE_BITS	(R_BASE_BYTES*8)	// size of base segment in number of bits
#define R_BASE_LAST	((r_base)1 << (R_BASE_BITS-1)) // mask of highest bit of base segment
#define R_BASE_MASK	((r_base)-1)	// mask of base segments (all bits are '1')

// number
#define R_BITS		(R_BYTES*8)		// total size in bits

// exponent
#define R_EXP_BYTES	(R_EXP_MAXBITS/8)	// size of exponent integer in bytes

 #ifdef R_COMP_BIAS	// compatible bias
#define R_EXP_BIAS	((r_exp)(((r_exp)1 << (R_EXP_BITS-1)) - 1)) // exponent bias (= exponent of number "1.0")
 #else
#define R_EXP_BIAS	((r_exp)(((r_exp)1 << (R_EXP_BITS-1)))) // exponent bias (= exponent of number "1.0")
 #endif

 #if R_EXP_MAXBITS == R_EXP_BITS
#define R_EXP_MASK	((r_exp)-1)		// exponent mask (all bits are '1')
 #else
#define R_EXP_MASK	((r_exp)(((r_exp)1 << R_EXP_BITS) - 1)) // exponent mask (all bits are '1')
 #endif

#define R_EXP_INF	R_EXP_MASK		// exponent infinity value
#define R_EXP_MAX	(R_EXP_INF-1)		// exponent maximal valid value
#define R_EXP_MIN	(r_exp)1		// exponent minimal valid value
#define R_EXP_0		(r_exp)0		// exponent special value - number is zero
#define R_EXP_1		(r_exp)R_EXP_BIAS	// exponent of number 1.0
#define R_EXP_SHIFT	(R_EXP_MAXBITS-1 - R_EXP_BITS) // shift exponent
#define R_DECEXP	(R_EXP_BITS-2)	// number of square decimal exponents

// mantissa
#define R_MANT_BITS	(R_BITS - R_EXP_BITS - 1) // size of mantissa in bits
#define R_MANT_BYTES ((R_MANT_BITS + 7)/8) // size of mantissa in number of bytes
#define R_MANT_NUM ((R_MANT_BITS + R_BASE_BITS - 1)/R_BASE_BITS) // size of mantissa in number of segments
#define R_MANT_LASTBITS_TMP (R_MANT_BITS - (R_MANT_BITS/R_BASE_BITS)*R_BASE_BITS) // number of bits of mantissa in last segment
  #if R_MANT_LASTBITS_TMP == 0 // mantissa fits into whole segments
#define R_MANT_MASK ((r_base)-1)		// mask of bits in last segment
#define R_MANT_LASTBITS R_BASE_BITS		// number of bits in last segment
#define R_MANT_WHOLENUM R_MANT_NUM		// number of whole segments
  #else
#define R_MANT_MASK (((r_base)1 << (R_MANT_LASTBITS_TMP)) - 1) // mask of bits in last segment
#define R_MANT_LASTBITS R_MANT_LASTBITS_TMP	// number of bits in last segment
#define R_MANT_WHOLENUM (R_MANT_NUM-1)	// number of whole segments
  #endif
#define R_MANT_LAST	((r_base)1 << (R_MANT_LASTBITS-1)) // mask of highest bit of last base segment
#define R_MANT_DIG	((R_MANT_BITS+1)*30103UL/100000) // max. number of digits per mantissa (= precision)

#define R_MAXROUND	(R_MANT_DIG+2)		// number of rounding corrections

#define R_TRIM		(R_MANT_BITS/16)	// number of bits to trim small differences

// ===========================================================================
//                        Memory number configuration
// ===========================================================================

// memory number
#if M_BYTES > R_BYTES
#error M_BYTES must not be larger than R_BYTES
#endif

#if M_EXP_BITS > R_EXP_BITS
#error M_EXP_BITS must not be larger than R_EXP_BITS
#endif

#define M_BITS		(M_BYTES*8)		// total size in bits

#ifdef M_COMP_BIAS
#define M_EXP_BIAS	((r_exp)(((r_exp)1 << (M_EXP_BITS-1)) - 1)) // exponent bias (= exponent of number "1.0")
 #else
#define M_EXP_BIAS	((r_exp)(((r_exp)1 << (M_EXP_BITS-1)))) // exponent bias (= exponent of number "1.0")
 #endif

 #if R_EXP_MAXBITS == M_EXP_BITS
#define M_EXP_MASK	((r_exp)-1)		// exponent mask (all bits are '1')
 #else
#define M_EXP_MASK	((r_exp)(((r_exp)1 << M_EXP_BITS) - 1)) // exponent mask (all bits are '1')
 #endif

#define M_EXP_INF	M_EXP_MASK		// exponent infinity value
#define M_EXP_MAX	(M_EXP_INF-1)		// exponent maximal valid value
#define M_EXP_MIN	(r_exp)1		// exponent minimal valid value
#define M_EXP_0		(r_exp)0		// exponent special value - number is zero
#define M_EXP_1		(r_exp)M_EXP_BIAS	// exponent of number 1.0
#define M_EXP_SHIFT	(R_EXP_MAXBITS-1 - M_EXP_BITS) // shift exponent
#define M_DECEXP	(M_EXP_BITS-2)	// number of square decimal exponents

#define M_MANT_BITS	(M_BITS - M_EXP_BITS - 1) // size of mantissa in bits

#if M_MANT_BITS > R_MANT_BITS
#error M_MANT_BITS must not be larger than R_MANT_BITS
#endif

#define M_MANT_BYTES ((M_MANT_BITS + 7)/8) // size of mantissa in number of bytes
#define M_MANT_NUM ((M_MANT_BITS + R_BASE_BITS - 1)/R_BASE_BITS) // size of mantissa in number of segments
#define M_MANT_LASTBITS_TMP (M_MANT_BITS - (M_MANT_BITS/R_BASE_BITS)*R_BASE_BITS) // number of bits of mantissa in last segment
  #if M_MANT_LASTBITS_TMP == 0 // mantissa fits into whole segments
#define M_MANT_MASK ((r_base)-1)		// mask of bits in last segment
#define M_MANT_LASTBITS R_BASE_BITS		// number of bits in last segment
#define M_MANT_WHOLENUM M_MANT_NUM		// number of whole segments
  #else
#define M_MANT_MASK (((r_base)1 << (M_MANT_LASTBITS_TMP)) - 1) // mask of bits in last segment
#define M_MANT_LASTBITS M_MANT_LASTBITS_TMP	// number of bits in last segment
#define M_MANT_WHOLENUM (M_MANT_NUM-1)	// number of whole segments
  #endif
#define M_MANT_LAST	((r_base)1 << (M_MANT_LASTBITS-1)) // mask of highest bit of last base segment
#define M_MANT_DIG	((M_MANT_BITS+1)*30103UL/100000) // max. number of digits per mantissa (= precision)

// ===========================================================================
//                           Real number structure
// ===========================================================================

// real number
typedef struct {
	u8	m[R_BYTES]; // mantissa (starting LSB), exponent, sign
} real;

// memory number
typedef struct {
	u8	m[M_BYTES]; // mantissa (starting LSB), exponent, sign
} mreal;

// initialize constants
void RealInit();

// ===========================================================================
//                 Low-level operations (mantissa and exponent)
// ===========================================================================

// get exponent
r_exp RealGetExp(const real* num);
r_exp MRealGetExp(const mreal* num);

// set exponent (must be in valid range)
void RealSetExp(real* num, r_exp exp);
void MRealSetExp(mreal* num, r_exp exp);

// get last segment of mantissa, may be partially
r_base RealGetLastSeg(const real* num);
r_base MRealGetLastSeg(const mreal* num);

// set last segment of mantissa
void RealSetLastSeg(real* num, r_base val);
void MRealSetLastSeg(mreal* num, r_base val);

// shift mantissa left to higher bits (returns highest carry bit)
u8 RealMantL(real* num, u8 carry);

// shift mantissa right to lower bits (returns lowest carry bit)
u8 RealMantR(real* num, u8 carry);

// add two mantissas (returns carry)
u8 RealMantAdd(real* dst, const real* src, u8 carry);

// subtract two mantissas (returns carry)
u8 RealMantSub(real* dst, const real* src, u8 carry);

// OR two mantissas
void RealMantOr(real* dst, const real* src);

// AND two mantissas
void RealMantAnd(real* dst, const real* src);

// XOR two mantissas
void RealMantXor(real* dst, const real* src);

// increment mantissa, returns carry bit (= result is 0)
u8 RealMantInc(real* num);

// invert mantissa
void RealMantNot(real* num);

// negate mantissa (returns carry if number is 0)
u8 RealMantNeg(real* num);

// precise truncation towards zero (no rounding epsilon delta, simple clears fractional bits)
void RealTruncPrec(real* num);

// ===========================================================================
//                      Flags and state manipulation
// ===========================================================================

// check if number is zero
Bool RealIsZero(const real* num);
Bool MRealIsZero(const mreal* num);

// check if number is infinity (positive or negative)
Bool RealIsInf(const real* num);

// check if number is negative
Bool RealIsNeg(const real* num);

// check if number is positive (but not zero)
Bool RealIsPos(const real* num);

// set sign (True = negative)
void RealSetSign(real* num, Bool sign);

// negate - flip sign (except zero)
void RealNeg(real* num);
void MRealNeg(mreal* num);

// absolute value - clear negative flag
void RealAbs(real* num);

// set negative flag (only if not 0)
void RealSetNeg(real* num);

// Signum - get number 0, 1 or -1, by sign
s8 RealGetSign(const real* num);

// Signum - set number to 0, 1 or -1, by sign
void RealSign(real* dst, const real* src);

// ===========================================================================
//                           Set number
// ===========================================================================

// clear number (exponent is hardcoded to 0)
void RealSet0(real* num);
void MRealSet0(mreal* num);

// set value 1
void RealSet1(real* num);

// set value -1
void RealSetM1(real* num);

// set value 2
void RealSet2(real* num);

// set value 0.5
void RealSet05(real* num);

// set minimal positive number
void RealSetMin(real* num);
void MRealSetMin(mreal* num);

// set maximal positive number
void RealSetMax(real* num);
void MRealSetMax(mreal* num);

// set positive infinity
void RealSetInf(real* num);
void MRealSetInf(mreal* num);

// set negative infinity
void RealSetMInf(real* num);
void MRealSetMInf(mreal* num);

// set unsigned integer value
void RealSetUInt(real* num, r_base n);
void RealSetU8(real* num, u8 n);
void RealSetU16(real* num, u16 n);
void RealSetU32(real* num, u32 n);

// set signed integer value
void RealSetSInt(real* num, r_bases n);
void RealSetS8(real* num, s8 n);
void RealSetS16(real* num, s16 n);
void RealSetS32(real* num, s32 n);

// get unsigned integer value (rounded towards zero)
r_base RealGetUInt(const real* num);
u8 RealGetU8(const real* num);
u16 RealGetU16(const real* num);
u32 RealGetU32(const real* num);

// get signed integer value (rounded towards zero)
r_bases RealGetSInt(const real* num);
s8 RealGetS8(const real* num);
s16 RealGetS16(const real* num);
s32 RealGetS32(const real* num);

// copy number
void RealCopy(real* dst, const real* src);

// exchange numbers
void RealExch(real* num1, real* num2);

// set constant from template in ROM
void RealConst(real* num, const real* temp);

// export into memory number
void RealSave(const real* num, mreal* mem);

// import from memory number
void RealLoad(real* num, const mreal* mem);

// ===========================================================================
//                          Arithmetics operations
// ===========================================================================

// add 2 numbers
void RealAdd(real* dst, const real* src1, const real* src2);

// subtract 2 numbers
void RealSub(real* dst, const real* src1, const real* src2);

// bitwise operation (oper = OPER_AND, OPER_OR, OPER_XOR)
void RealBit(real* dst, const real* src1, const real* src2, u8 oper);

// multiply 2 numbers
void RealMul(real* dst, const real* src1, const real* src2);

// divide 2 numbers
void RealDiv(real* dst, const real* src1, const real* src2);

// power of 2 numbers (y = base^exp)
void RealPow(real* dst, const real* base, const real* exp);

// root of 2 numbers (y = Vbase^exp)
void RealRoot(real* dst, const real* base, const real* exp);

// modulus (with floor rounding; result has same sign as divisor src2)
void RealModFloor(real* dst, const real* src1, const real* src2);

// modulus (with trunc rounding; result has same sign as dividend src1)
void RealModTrunc(real* dst, const real* src1, const real* src2);

// compare 2 numbers (returns -1: num1<num2, 0: num1=num2, +1: num1>num2)
s8 RealComp(const real* num1, const real* num2);

// ===========================================================================
//                             Functions
// ===========================================================================

// compare number to zero (returns -1: num<0, 0: num=0, +1: num>0)
s8 RealComp0(const real* num);

// square x^2
void RealSqr(real* dst, const real* src);

// reciprocal value 1/x
void RealRec(real* dst, const real* src);

// multiply by 2 (increases exponent by 1)
void RealMul2(real* num);

// divide by 2 (decreases exponent by 1)
void RealDiv2(real* num);

// shift left (multiply by power of 2)
void RealLeft(real* num, r_exp n);

// shift right (divide by power of 2)
void RealRight(real* num, r_exp n);

// bitwise NOT
void RealNot(real* num);

// increase number by 1
void RealInc(real* dst, const real* src);

// decrease number by 1
void RealDec(real* dst, const real* src);

// rounding to nearest (returns sign of difference and rounding -1 up, 0 or +1 down)
//  num = source number
//  integ = output integer number, can be NULL
//  dif = difference num-integ = -0.5..+0.5, can be NULL
s8 RealRound(const real* num, real* integ, real* frac);

// truncation towards zero (= "integer" function, returns True if integer)
//  num = source number
//  integ = output integer number, can be NULL
//  dif = difference num-integ = -1..+1, can be NULL
Bool RealTrunc(const real* num, real* integ, real* frac);

// rounding down (returns True if integer)
//  num = source number
//  integ = output integer number, can be NULL
//  dif = difference num-integ = 0..+1, can be NULL
Bool RealFloor(const real* num, real* integ, real* frac);

// rounding up (returns True if integer)
//  num = source number
//  integ = output integer number, can be NULL
//  dif = difference num-integ = -1..0, can be NULL
Bool RealCeil(const real* num, real* integ, real* frac);

// check if number is integer
Bool RealIsInt(const real* num);

// natural logarithm
void RealLn(real* num);

// ln(2) - calculates ln(2) constant
void RealLn2(real* num);

// natural exponent
void RealExp(real* num);

// decimal logarithm
void RealLog10(real* num);

// decimal exponent
void RealExp10(real* num);

// binary logarithm
void RealLog2(real* num);

// binary exponent
void RealExp2(real* num);

// sqrt
void RealSqrt(real* num);

// convert angle from current angle unit to radians
void RealToRad(real* num);

// convert angle from radians to current angle unit
void RealFromRad(real* num);

// normalize angle in radians into range 0..PI*2 (= 0..360)
void RealNormRad(real* num);

// sine
void RealSin(real* num);

// cosine
void RealCos(real* num);

// tangent
void RealTan(real* num);

// cotangent
void RealCoTan(real* num);

// arcus sine (result is in range -PI/2..+PI/2)
void RealASin(real* num);

// arcus cosine (result is in range 0..PI)
void RealACos(real* num);

// arcus tangent (result is in range -PI/2..+PI/2)
void RealATan(real* num);

// arcus cotangent (result is in range -PI/2..+PI/2)
void RealACoTan(real* num);

// hyperbolic sine
void RealSinH(real* num);

// hyperbolic cosine
void RealCosH(real* num);

// hyperbolic tangent
void RealTanH(real* num);

// hyperbolic cotangent
void RealCoTanH(real* num);

// hyperbolic secant
void RealSecH(real* num);

// hyperbolic cosecant
void RealCscH(real* num);

// areasine, inverse hyperbolic sine
void RealArSinH(real* num);

// areacosine, inverse hyperbolic cosine
void RealArCosH(real* num);

// areatangent, inverse hyperbolic tangent
void RealArTanH(real* num);

// areacotangent, inverse hyperbolic cotangent
void RealArCoTanH(real* num);

// areasecant, inverse hyperbolic secant
void RealArSecH(real* num);

// areacosecant, inverse hyperbolic cosecant
void RealArCscH(real* num);

// set seed of random generator
void SetRndSeed(u32 seed);

// get seed of random generator
u32 GetRndSeed();

// shift random generator
u32 RndShift();

// set random number in range 0<= .. <1 (32-bit precision)
void RealSetRnd(real* num);

// random number in range 0<= .. <num (32-bit precision)
void RealRnd(real* num);

// integer factorial
void RealFactInt(real* num, u16 n);

// non-integer factorial approximation (precission 27 digits or more)
void RealFact(real* num);

// natural logarithm of factorial (using approximation)
void RealFactLn(real* num);

// decimal logarithm of factorial (using approximation)
void RealFactLog(real* num);

// check precision of factorial approximation - get number of valid digits of RealFactSmooth
//void RealFactCheck(real* num, u16 n);

// ===========================================================================
//                             Constants
// ===========================================================================

#define FACT_COEFF	12		// number of factorial coefficients

// constant in RAM
extern real RealConst0;		// 0
#define RealConst1 RealConstExp[R_DECEXP] // 1
extern real RealConstM1;	// -1
#define RealConst05 RealConstRound[0] // 0.5
extern real RealConst2;		// 2
#define RealConst10 RealConstExp[R_DECEXP+1] // 10
extern real RealConst075;	// 0.75
#define RealConst100 RealConstExp[R_DECEXP+2] // 100

// pre-calculated constants
#define RealConst01 RealConstExp[R_DECEXP-1] // 0.1
#define RealConst001 RealConstExp[R_DECEXP-2] // 0.01
#define RealConst00001 RealConstExp[R_DECEXP-3] // 0.0001
#define RealConst000000001 RealConstExp[R_DECEXP-4] // 0.00000001
extern real RealConstLn2;	// ln(2)
extern real RealConstRLn2;	// 1/ln(2)
extern real RealConstLn10;	// ln(10)
extern real RealConstRLn10; // 1/ln(10)
extern real RealConstLog2;	// log(2)
extern real RealConstExpMax; // max. exp(x)
extern real RealConstExpMin; // min. exp(x)
extern real RealConstEul;	// Eul
extern real RealConstPi05; // pi/2
extern real RealConstPi; // pi
extern real RealConstPi2; // pi*2
extern real RealConstLnPi22; // ln(pi*2)/2
extern real RealConst180Pi; // 180/pi
extern real RealConstPi180; // pi/180
extern real RealConst200Pi; // 200/pi
extern real RealConstPi200; // pi/200

extern real RealConstExp[R_DECEXP*2+1]; // decimal exponents (index R_DECEXP = number '1')
extern real RealConstRound[R_MANT_DIG+2]; // rounding corrections (0.5, 0.05,...)
extern real RealConstFactA[FACT_COEFF]; // factorial coefficients

// template of constants in ROM
extern const real RealTempLn2 PROGMEM;	// ln(2)
extern const real RealTempRLn2 PROGMEM; // 1/ln(2)
extern const real RealTempLn10 PROGMEM; // ln(10)
extern const real RealTempRLn10 PROGMEM; // 1/ln(10)
extern const real RealTempLog2 PROGMEM; // log(2)
extern const real RealTempExpMax PROGMEM; // max. exp(x)
extern const real RealTempExpMin PROGMEM; // min. exp(x)
extern const real RealTempEul PROGMEM;	// Eul
extern const real RealTempPi05 PROGMEM; // pi/2
extern const real RealTempPi PROGMEM; // pi
extern const real RealTempPi2 PROGMEM; // pi*2
extern const real RealTempLnPi22 PROGMEM; // ln(pi*2)/2
extern const real RealTemp180Pi PROGMEM; // 180/pi
extern const real RealTempPi180 PROGMEM; // pi/180
extern const real RealTemp200Pi PROGMEM; // 200/pi
extern const real RealTempPi200 PROGMEM; // pi/200

extern const real RealTempExp[R_DECEXP*2+1] PROGMEM; // decimal exponents (index R_DECEXP = number '1')
extern const real RealTempRound[R_MANT_DIG+2] PROGMEM; // rounding corrections (0.5, 0.05,...)
extern const real RealTempFactA[FACT_COEFF] PROGMEM; // factorial coefficients

// random seed
extern u32 RandSeed;
