
// ****************************************************************************
//                                 
//                         Real numbers (floating-point)
//
// ****************************************************************************

#include "include.h"

// current angle unit (UNIT_*)
u8 Unit;

// constant in RAM (4) ... constants in RAM total 87 (870 B)
real RealConst0;	// 0
real RealConstM1;	// -1
real RealConst2;	// 2
real RealConst075;	// 0.75
real RealConstMore05;	// a little above 0.5 (used by RealRound function)

// pre-calculated constants (16)
real RealConstLn2;	// ln(2)
real RealConstRLn2;	// 1/ln(2)
real RealConstLn10;	// ln(10)
real RealConstRLn10; // 1/ln(10)
real RealConstLog2;	// log(2)
real RealConstExpMax; // max. exp(x)
real RealConstExpMin; // min. exp(x)
real RealConstEul;	// Eul
real RealConstPi05; // pi/2
real RealConstPi; // pi
real RealConstPi2; // pi*2
real RealConstLnPi22; // ln(pi*2)/2
real RealConst180Pi; // 180/pi
real RealConstPi180; // pi/180
real RealConst200Pi; // 200/pi
real RealConstPi200; // pi/200

real RealConstExp[R_DECEXP*2+1]; // decimal exponents (index R_DECEXP = number '1') (33)
real RealConstRound[R_MANT_DIG+2]; // rounding corrections (0.5, 0.05,...) (double: 23)
real RealConstFactA[FACT_COEFF]; // factorial coefficients (12)

// random seed
u32 RandSeed;

#ifdef GENCONST			// generate constants

// factorial coefficients templates
const char* FactA[FACT_COEFF] PROGMEM = {
		"1",	// a0
		"1",	// a1
		"53",	// a2
		"195",	// a3
		"22999",	// a4
		"29944523",	// a5
		"109535241009",	// a6
		"29404527905795295658",	// a7
		"455377030420113432210116914702",	// a8
		"26370812569397719001931992945645578779849",	// a9
		"152537496709054809881638897472985990866753853122697839",	// a10
		"100043420063777451042472529806266909090824649341814868347109676190691",	// a11
};

const char* FactB[FACT_COEFF] PROGMEM = {
		"12",	// a0
		"30",	// a1
		"210",	// a2
		"371",	// a3
		"22737",	// a4
		"19733142",	// a5
		"48264275462",	// a6
		"9769214287853155785",	// a7
		"113084128923675014537885725485",	// a8
		"5271244267917980801966553649147604697542", // a9
		"24274291553105128438297398108902195365373879212227726",	// a10
		"13346384670164266280033479022693768890138348905413621178450736182873",	// a11
};

#endif // GENCONST

// initialize real numbers
void RealInit()
{
	int i;

#ifdef GENCONST			// generate constants
	int i2;
	real m;
#endif

	RealSet0(&RealConst0); // 0
	RealSet1(&RealConst1); // 1
	RealSetM1(&RealConstM1); // -1
	RealSet05(&RealConst05); // 0.5
	RealSet2(&RealConst2); // 2
	RealSetU8(&RealConst10, 10); // 10
	RealSetU8(&RealConst075, 3); RealDiv2(&RealConst075); RealDiv2(&RealConst075); // 0.75

	RealSet05(&RealConstMore05); RealConstMore05.m[0] = 5;	// a little above 0.5 (used by RealRound function)

#ifndef GENCONST			// generate constants

	// load constants
	RealConst(&RealConstLn2, &RealTempLn2);	// ln(2)
	RealConst(&RealConstRLn2, &RealTempRLn2); // 1/ln(2)
	RealConst(&RealConstLn10, &RealTempLn10); // ln(10)
	RealConst(&RealConstRLn10, &RealTempRLn10); // 1/ln(10)
	RealConst(&RealConstLog2, &RealTempLog2); // log(2)
	RealConst(&RealConstExpMax, &RealTempExpMax); // max. exp(x)
	RealConst(&RealConstExpMin, &RealTempExpMin); // min. exp(x)
	RealConst(&RealConstEul, &RealTempEul);	// Eul
	RealConst(&RealConstPi05, &RealTempPi05); // pi/2
	RealConst(&RealConstPi, &RealTempPi); // pi
	RealConst(&RealConstPi2, &RealTempPi2); // pi*2
	RealConst(&RealConstLnPi22, &RealTempLnPi22); // ln(pi*2)/2
	RealConst(&RealConst180Pi, &RealTemp180Pi); // 180/pi
	RealConst(&RealConstPi180, &RealTempPi180); // pi/180
	RealConst(&RealConst200Pi, &RealTemp200Pi); // 200/pi
	RealConst(&RealConstPi200, &RealTempPi200); // pi/200

	for (i = 0; i < R_DECEXP*2+1; i++)
		RealConst(&RealConstExp[i], &RealTempExp[i]); // decimal exponents (index R_DECEXP = number '1') (33)

	for (i = 0; i < R_MANT_DIG+2; i++)
		RealConst(&RealConstRound[i], &RealTempRound[i]); // rounding corrections (0.5, 0.05,...) (double: 23)

	for (i = 0; i < FACT_COEFF; i++)
		RealConst(&RealConstFactA[i], &RealTempFactA[i]); // factorial coefficients (12)

#else // GENCONST

	// calculate constants
	RealDiv(&RealConst01, &RealConst1, &RealConst10); // 0.1
	RealLn2(&RealConstLn2); // ln(2)
	RealDiv(&RealConstRLn2, &RealConst1, &RealConstLn2); // 1/ln(2)
	RealSetU8(&RealConstLn10, 10); RealLn(&RealConstLn10); // ln(10)
	RealDiv(&RealConstRLn10, &RealConst1, &RealConstLn10); // 1/ln(10)
	RealDiv(&RealConstLog2, &RealConstLn2, &RealConstLn10); // log(2)
	RealSetMax(&RealConstExpMax); RealLn(&RealConstExpMax); // max. exp(x)
	RealSetMin(&RealConstExpMin); RealLn(&RealConstExpMin); // min. exp(x)
	RealSet1(&RealConstEul); RealExp(&RealConstEul); // Eul
	RealCopy(&RealConstPi05, &RealConst05); RealASin(&RealConstPi05); RealSetU8(&RealConstPi, 3); RealMul(&RealConstPi05, &RealConstPi05, &RealConstPi); // pi/2
	RealCopy(&RealConstPi, &RealConstPi05); RealMul2(&RealConstPi);
	RealCopy(&RealConstPi2, &RealConstPi); RealMul2(&RealConstPi2);
	RealCopy(&RealConstLnPi22, &RealConstPi2); RealLn(&RealConstLnPi22); RealDiv2(&RealConstLnPi22);
	RealSetU8(&RealConst180Pi, 180); RealDiv(&RealConst180Pi, &RealConst180Pi, &RealConstPi);
	RealSetU8(&RealConstPi180, 180); RealDiv(&RealConstPi180, &RealConstPi, &RealConstPi180);
	RealSetU8(&RealConst200Pi, 200); RealDiv(&RealConst200Pi, &RealConst200Pi, &RealConstPi);
	RealSetU8(&RealConstPi200, 200); RealDiv(&RealConstPi200, &RealConstPi, &RealConstPi200);

	// decimal exponent
	i2 = R_DECEXP-2;
	for (i = R_DECEXP+2; i < R_DECEXP*2+1; i++)
	{
		RealMul(&RealConstExp[i], &RealConstExp[i-1], &RealConstExp[i-1]);
		RealDiv(&RealConstExp[i2], &RealConst1, &RealConstExp[i]);
		i2--;
	}

	// rounding corrections
	for (i = 1; i < R_MANT_DIG+2; i++) RealMul(&RealConstRound[i], &RealConstRound[i-1], &RealConst01);

	// factorial coefficients
	for (i = 0; i < FACT_COEFF; i++)
	{
		RealFromText(&RealConstFactA[i], FactA[i], True);
		RealFromText(&m, FactB[i], True);
		RealDiv(&RealConstFactA[i], &RealConstFactA[i], &m);
	}

#endif // GENCONST
}

// ===========================================================================
//                 Low-level operations (mantissa and exponent)
// ===========================================================================

// get exponent
r_exp RealGetExp(const real* num)
{
	r_exp exp = *(r_exp*)&num->m[R_BYTES-R_EXP_BYTES]; // get exponent

#if R_EXP_MAXBITS == R_EXP_BITS // using whole exponent, we need 1 extra lower bit
	exp <<= 1; // destroy sign bit
	if ((num->m[R_BYTES-R_EXP_BYTES-1] & 0x80) != 0) exp |= 1; // add 1 extra bit
#else
	exp >>= R_EXP_SHIFT; // shift exponent to right position
	exp &= R_EXP_MASK; // mask exponent
#endif
	return exp;
}

r_exp MRealGetExp(const mreal* num)
{
	r_exp exp = *(r_exp*)&num->m[M_BYTES-R_EXP_BYTES]; // get exponent

#if R_EXP_MAXBITS == M_EXP_BITS // using whole exponent, we need 1 extra lower bit
	exp <<= 1; // destroy sign bit
	if ((num->m[M_BYTES-R_EXP_BYTES-1] & 0x80) != 0) exp |= 1; // add 1 extra bit
#else
	exp >>= M_EXP_SHIFT; // shift exponent to right position
	exp &= M_EXP_MASK; // mask exponent
#endif
	return exp;
}

// set exponent (must be in valid range)
void RealSetExp(real* num, r_exp exp)
{
	r_exp tmpexp = *(r_exp*)&num->m[R_BYTES-R_EXP_BYTES]; // get exponent
#if R_EXP_MAXBITS == R_EXP_BITS // we need to set 1 extra lower bit
	tmpexp &= ~(R_EXP_MASK >> 1); // clear current exponent (leave only sign bit)
	num->m[R_BYTES-R_EXP_BYTES-1] &= 0x7f; // clear 1 extra lower bit
	tmpexp |= exp >> 1; // add new exponent
	if ((exp & 1) != 0) num->m[R_BYTES-R_EXP_BYTES-1] |= 0x80; // set extra lower bit
#else
	tmpexp &= ~(R_EXP_MASK << R_EXP_SHIFT); // clear current exponent
	tmpexp |= exp << R_EXP_SHIFT; // add new exponent
#endif
	*(r_exp*)&num->m[R_BYTES-R_EXP_BYTES] = tmpexp; // set exponent
}

void MRealSetExp(mreal* num, r_exp exp)
{
	r_exp tmpexp = *(r_exp*)&num->m[M_BYTES-R_EXP_BYTES]; // get exponent
#if R_EXP_MAXBITS == M_EXP_BITS // we need to set 1 extra lower bit
	tmpexp &= ~(M_EXP_MASK >> 1); // clear current exponent (leave only sign bit)
	num->m[M_BYTES-R_EXP_BYTES-1] &= 0x7f; // clear 1 extra lower bit
	tmpexp |= exp >> 1; // add new exponent
	if ((exp & 1) != 0) num->m[M_BYTES-R_EXP_BYTES-1] |= 0x80; // set extra lower bit
#else
	tmpexp &= ~(M_EXP_MASK << M_EXP_SHIFT); // clear current exponent
	tmpexp |= exp << M_EXP_SHIFT; // add new exponent
#endif
	*(r_exp*)&num->m[M_BYTES-R_EXP_BYTES] = tmpexp; // set exponent
}

// get last segment of mantissa, may be partially
r_base RealGetLastSeg(const real* num)
{
	r_base seg = *(r_base*)&num->m[(R_MANT_NUM-1)*R_BASE_BYTES];
#if R_MANT_LASTBITS != R_BASE_BITS
	seg &= R_MANT_MASK;
#endif
	return seg;
}

r_base MRealGetLastSeg(const mreal* num)
{
	r_base seg = *(r_base*)&num->m[(M_MANT_NUM-1)*R_BASE_BYTES];
#if M_MANT_LASTBITS != R_BASE_BITS
	seg &= M_MANT_MASK;
#endif
	return seg;
}

// set last segment of mantissa
void RealSetLastSeg(real* num, r_base val)
{
#if R_MANT_LASTBITS != R_BASE_BITS
	r_base seg;
	seg = *(r_base*)&num->m[(R_MANT_NUM-1)*R_BASE_BYTES];
	seg &= ~R_MANT_MASK;
	val &= R_MANT_MASK;
	val |= seg;
#endif
	*(r_base*)&num->m[(R_MANT_NUM-1)*R_BASE_BYTES] = val;
}

void MRealSetLastSeg(mreal* num, r_base val)
{
#if M_MANT_LASTBITS != R_BASE_BITS
	r_base seg;
	seg = *(r_base*)&num->m[(M_MANT_NUM-1)*R_BASE_BYTES];
	seg &= ~M_MANT_MASK;
	val &= M_MANT_MASK;
	val |= seg;
#endif
	*(r_base*)&num->m[(M_MANT_NUM-1)*R_BASE_BYTES] = val;
}

// shift mantissa left to higher bits (returns highest carry bit)
u8 RealMantL(real* num, u8 carry)
{
	int i;
	r_base b, *s;
	u8 carry2;

	s = (r_base*)num->m;

	// shift whole segments
	for (i = R_MANT_WHOLENUM; i > 0; i--)
	{
		b = *s;
		carry2 = carry;
		carry = ((b & R_BASE_LAST) == 0) ? 0 : 1; // carry to higher segment
		b = (b << 1) | carry2; // shift and add new carry
		*s++ = b; // set segment
	}

	// shift last partially segment
#if R_MANT_LASTBITS != R_BASE_BITS
	b = RealGetLastSeg(num);
	carry2 = carry;
	carry = ((b & R_MANT_LAST) == 0) ? 0 : 1; // carry to higher segment
	b = (b << 1) | carry2; // shift and add new carry
	RealSetLastSeg(num, b);
#endif

	return carry;
}

// shift mantissa right to lower bits (returns lowest carry bit)
u8 RealMantR(real* num, u8 carry)
{
	int i;
	r_base b, *s;
	u8 carry2;

	// shift last partially segment
#if R_MANT_LASTBITS != R_BASE_BITS
	b = RealGetLastSeg(num);
	carry2 = carry;
	carry = (u8)(b & 1); // carry to lower segment
	b >>= 1; // shift
	if (carry2) b |= R_MANT_LAST; // add cary from higher segment
	RealSetLastSeg(num, b);
#endif

	// shift whole segments
	s = &((r_base*)num->m)[R_MANT_WHOLENUM-1];
	for (i = R_MANT_WHOLENUM; i > 0; i--)
	{
		b = *s;
		carry2 = carry;
		carry = (u8)(b & 1); // carry to lower segment
		b >>= 1; // shift
		if (carry2) b |= R_BASE_LAST; // add cary from higher segment
		*s-- = b; // set segment
	}

	return carry;
}

// add two mantissas (returns carry)
u8 RealMantAdd(real* dst, const real* src, u8 carry)
{
	r_base* d;
	const r_base* s;
	r_base a, b;
	int i;
	s = (const r_base*)src->m;
	d = (r_base*)dst->m;

	// add whole segments
	for (i = R_MANT_WHOLENUM; i > 0; i--)
	{
		a = *d + carry;
		carry = (a < carry) ? 1 : 0;
		b = *s;
		a += b;
		carry += (a < b) ? 1 : 0;
		*d = a;
		d++;
		s++;
	}

	// add last partially segments
#if R_MANT_LASTBITS != R_BASE_BITS
	a = RealGetLastSeg(dst);
	a += carry;
	carry = 0;
	if (a > R_MANT_MASK)
	{
		a &= R_MANT_MASK;
		carry = 1;
	}
	b = RealGetLastSeg(src);
	a += b;
	if (a > R_MANT_MASK)
	{
		a &= R_MANT_MASK;
		carry = 1;
	}
	RealSetLastSeg(dst, a);
#endif

	return carry;
}

// subtract two mantissas (returns carry)
u8 RealMantSub(real* dst, const real* src, u8 carry)
{
	r_base* d;
	const r_base* s;
	r_base a, b;
	int i;
	s = (const r_base*)src->m;
	d = (r_base*)dst->m;

	// subtract whole segments
	for (i = R_MANT_WHOLENUM; i > 0; i--)
	{
		b = *s + carry;
		carry = (b < carry) ? 1 : 0;
		a = *d;
		carry += (a < b) ? 1 : 0;
		*d = a - b;
		d++;
		s++;
	}

	// subtract last partially segments
#if R_MANT_LASTBITS != R_BASE_BITS
	b = RealGetLastSeg(src);
	b += carry;
	carry = 0;
	if (b > R_MANT_MASK)
	{
		b &= R_MANT_MASK;
		carry = 1;
	}
	a = RealGetLastSeg(dst);
	carry += (a < b) ? 1 : 0;
	a = a - b;
	RealSetLastSeg(dst, a);
#endif

	return carry;
}

// OR two mantissas
void RealMantOr(real* dst, const real* src)
{
	r_base* d;
	const r_base* s;
	int i;
	s = (const r_base*)src->m;
	d = (r_base*)dst->m;

	// OR whole segments
	for (i = R_MANT_WHOLENUM; i > 0; i--)
	{
		*d = *d | *s;
		d++;
		s++;
	}

	// OR last partially segments
#if R_MANT_LASTBITS != R_BASE_BITS
	RealSetLastSeg(dst, RealGetLastSeg(dst) | RealGetLastSeg(src));
#endif
}

// AND two mantissas
void RealMantAnd(real* dst, const real* src)
{
	r_base* d;
	const r_base* s;
	int i;
	s = (const r_base*)src->m;
	d = (r_base*)dst->m;

	// OR whole segments
	for (i = R_MANT_WHOLENUM; i > 0; i--)
	{
		*d = *d & *s;
		d++;
		s++;
	}

	// OR last partially segments
#if R_MANT_LASTBITS != R_BASE_BITS
	RealSetLastSeg(dst, RealGetLastSeg(dst) & RealGetLastSeg(src));
#endif
}

// XOR two mantissas
void RealMantXor(real* dst, const real* src)
{
	r_base* d;
	const r_base* s;
	int i;
	s = (const r_base*)src->m;
	d = (r_base*)dst->m;

	// OR whole segments
	for (i = R_MANT_WHOLENUM; i > 0; i--)
	{
		*d = *d ^ *s;
		d++;
		s++;
	}

	// OR last partially segments
#if R_MANT_LASTBITS != R_BASE_BITS
	RealSetLastSeg(dst, RealGetLastSeg(dst) ^ RealGetLastSeg(src));
#endif
}

// increment mantissa, returns carry bit (= result is 0)
u8 RealMantInc(real* num)
{
	int i;
	r_base b, *s;

	s = (r_base*)num->m;

	// increment whole segments
	for (i = R_MANT_WHOLENUM; i > 0; i--)
	{
		b = *s + 1;
		*s++ = b;
		if (b != 0) return 0; // no carry
	}

	// increment last partially segment
#if R_MANT_LASTBITS != R_BASE_BITS
	b = RealGetLastSeg(num) + 1;
	RealSetLastSeg(num, b);
	if ((b & R_MANT_MASK) != 0) return 0; // no carry
#endif

	// carry
	return 1;
}

// invert mantissa
void RealMantNot(real* num)
{
	r_base *s;
	int i;

	s = (r_base*)num->m;

	// invert whole segments
	for (i = R_MANT_WHOLENUM; i > 0; i--)
	{
		*s = ~*s;
		s++;
	}

	// invert last partially segment
#if R_MANT_LASTBITS != R_BASE_BITS
	RealSetLastSeg(num, ~RealGetLastSeg(num));
#endif
}

// negate mantissa (returns carry if number is 0)
u8 RealMantNeg(real* num)
{
	// invert and then increment
	RealMantNot(num);
	return RealMantInc(num);
}

// precise truncation towards zero (no rounding epsilon delta, simple clears fractional bits)
void RealTruncPrec(real* num)
{
	int i;
	r_base *s;
	r_exp exp;

	// get exponent
	exp = RealGetExp(num);

	// small number < 1 can be made zero
	if (exp < R_EXP_1)
	{
		RealSet0(num);
		return;
	}

	// big number is always integer
	exp -= R_EXP_1;
	if (exp >= R_MANT_BITS) return;

	// prepare number of bits to clear (range 1..R_MANT_BITS)
	i = R_MANT_BITS - exp;

	// reset whole segments
	s = (r_base*)num->m;
	for (; i >= R_BASE_BITS; i -= R_BASE_BITS) *s++ = 0;

	// reset remaining bits
	if (i > 0) *s = *s & (R_BASE_MASK << i);
}

// ===========================================================================
//                      Flags and state manipulation
// ===========================================================================

// check if number is zero
Bool RealIsZero(const real* num)
{
	return RealGetExp(num) == R_EXP_0;
}

Bool MRealIsZero(const mreal* num)
{
	return MRealGetExp(num) == M_EXP_0;
}

// check if number is infinity (positive or negative)
Bool RealIsInf(const real* num)
{
	return RealGetExp(num) == R_EXP_INF;
}

// check if number is negative
Bool RealIsNeg(const real* num)
{
	return (num->m[R_BYTES-1] & 0x80) != 0;
}

// check if number is positive (but not zero)
Bool RealIsPos(const real* num)
{
	return !RealIsNeg(num) && !RealIsZero(num);
}

// set sign (True = negative)
void RealSetSign(real* num, Bool sign)
{
	u8 k = num->m[R_BYTES-1] & 0x7f;
	if (sign) k |= 0x80;
	num->m[R_BYTES-1] = k;
}

// negate - flip sign (except zero)
void RealNeg(real* num)
{
	if (!RealIsZero(num)) num->m[R_BYTES-1] ^= 0x80;
}

void MRealNeg(mreal* num)
{
	if (!MRealIsZero(num)) num->m[M_BYTES-1] ^= 0x80;
}

// absolute value - clear negative flag
void RealAbs(real* num)
{
	num->m[R_BYTES-1] &= 0x7f;
}

// set negative flag (only if not 0)
void RealSetNeg(real* num)
{
	if (!RealIsZero(num)) num->m[R_BYTES-1] |= 0x80;
}

// Signum - get number 0, 1 or -1, by sign
s8 RealGetSign(const real* num)
{
	if (RealIsZero(num)) return 0;
	if (RealIsNeg(num)) return -1;
	return 1;
}

// Signum - set number to 0, 1 or -1, by sign
void RealSign(real* dst, const real* src)
{
	if (RealIsZero(src))
	{
		RealSet0(dst);
	}
	else
	{
		if (RealIsNeg(src))
			RealSetM1(dst);
		else
			RealSet1(dst);
	}
}

// ===========================================================================
//                           Set number
// ===========================================================================

// clear number (exponent is hardcoded to 0)
void RealSet0(real* num)
{
	int i;
	u8* d = num->m;
	for (i = R_BYTES; i > 0; i--) *d++ = 0;
}

void MRealSet0(mreal* num)
{
	int i;
	u8* d = num->m;
	for (i = M_BYTES; i > 0; i--) *d++ = 0;
}

// set value 1
void RealSet1(real* num)
{
	RealSet0(num);
	RealSetExp(num, R_EXP_1);
}

// set value -1
void RealSetM1(real* num)
{
	RealSet1(num);
	RealNeg(num);
}

// set value 2
void RealSet2(real* num)
{
	RealSet0(num);
	RealSetExp(num, R_EXP_1 + 1);
}

// set value 0.5
void RealSet05(real* num)
{
	RealSet0(num);
	RealSetExp(num, R_EXP_1 - 1);
}

// set minimal positive number
void RealSetMin(real* num)
{
	RealSet0(num);
	RealSetExp(num, R_EXP_MIN);
}

void MRealSetMin(mreal* num)
{
	int i;
	u8* d = num->m;
	for (i = M_BYTES; i > 0; i--) *d++ = 0;
	MRealSetExp(num, M_EXP_MIN);
}

// set maximal positive number
void RealSetMax(real* num)
{
	int i;
	u8* d = num->m;
	for (i = R_BYTES-1; i > 0; i--) *d++ = 0xff;
	*d++ = 0x7f;

	RealSetExp(num, R_EXP_MAX);
}

void MRealSetMax(mreal* num)
{
	int i;
	u8* d = num->m;
	for (i = M_BYTES-1; i > 0; i--) *d++ = 0xff;
	*d++ = 0x7f;

	MRealSetExp(num, M_EXP_MAX);
}

// set positive infinity
void RealSetInf(real* num)
{
	RealSet0(num);
	RealSetLastSeg(num, R_MANT_LAST); // mantissa 1.5, middle of interval
	RealSetExp(num, R_EXP_INF);
}

void MRealSetInf(mreal* num)
{
	MRealSet0(num);
	MRealSetLastSeg(num, M_MANT_LAST); // mantissa 1.5, middle of interval
	MRealSetExp(num, M_EXP_INF);
}

// set negative infinity
void RealSetMInf(real* num)
{
	RealSetInf(num);
	RealNeg(num);
}

void MRealSetMInf(mreal* num)
{
	MRealSetInf(num);
	MRealNeg(num);
}

// set unsigned integer value
void RealSetUInt(real* num, r_base n)
{
	r_exp exp;

	// clear number
	RealSet0(num);

	// number is zero
	if (n == 0) return;

	// normalize number to highest bits
	exp = R_EXP_1 + R_BASE_BITS - 1; // exponent for case of max. number
	while ((n & R_BASE_LAST) == 0)
	{
		n <<= 1;
		exp--;
	}

	// delete hidden bit
	n <<= 1;

	// save whole last segment
#if R_BASE_BITS == R_MANT_LASTBITS
	((r_base*)num->m)[R_MANT_NUM-1] = n;
#else
	// save number into last segment
	RealSetLastSeg(num, n >> (R_BASE_BITS - R_MANT_LASTBITS));

	// save number into previous segment
 #if R_MANT_WHOLENUM > 0
	((r_base*)num->m)[R_MANT_WHOLENUM-1] = n << R_MANT_LASTBITS;
 #endif

#endif

	// save exponent
	RealSetExp(num, exp);
}

void RealSetU8(real* num, u8 n)
{
	RealSetUInt(num, n);
}

void RealSetU16(real* num, u16 n)
{
#if R_BASE_BYTES >= 2 // size of segment is "word" or more

	RealSetUInt(num, n);

#else // R_BASE_BYTES >= 2
// size of segment is "byte"

	r_exp exp;

	// clear number
	RealSet0(num);

	// number is zero
	if (n == 0) return;

	// normalize number to highest bits
	exp = R_EXP_1 + 16 - 1; // exponent for case of max. number
	while ((n & 0x8000) == 0)
	{
		n <<= 1;
		exp--;
	}

	// delete hidden bit
	n <<= 1;

	// save whole segments 8 bit
 #if R_BASE_BITS == R_MANT_LASTBITS

	num->m[R_MANT_NUM-1] = (u8)(n >> 8);
  #if R_MANT_NUM > 1
	num->m[R_MANT_NUM-2] = (u8)n;
  #endif

 #else // R_BASE_BITS == R_MANT_LASTBITS
	// save shifted segments 8 bit
	RealSetLastSeg(num, (u8)(n >> (16 - R_MANT_LASTBITS)));

  #if R_MANT_NUM > 1
	num->m[R_MANT_NUM-2] = (u8)(n >> (8 - R_MANT_LASTBITS));
  #endif

  #if R_MANT_NUM > 2
	num->m[R_MANT_NUM-3] = (u8)(n << R_MANT_LASTBITS);
  #endif

 #endif // R_BASE_BITS == R_MANT_LASTBITS

	// save exponent
	RealSetExp(num, exp);

#endif // R_BASE_BYTES >= 2
}

void RealSetU32(real* num, u32 n)
{
#if R_BASE_BYTES >= 4 // size of segment is "dword" or more

	RealSetUInt(num, n);

#else // R_BASE_BYTES >= 4

// size is "byte" or "word"

	r_exp exp;

	// clear number
	RealSet0(num);

	// number is zero
	if (n == 0) return;

	// normalize number to highest bits
	exp = R_EXP_1 + 32 - 1; // exponent for case of max. number
	while ((n & 0x80000000UL) == 0)
	{
		n <<= 1;
		exp--;
	}

	// delete hidden bit
	n <<= 1;

 #if R_BASE_BITS == 8

// segment is 8 bits

	// save whole segments 8 bit
  #if R_BASE_BITS == R_MANT_LASTBITS

	num->m[R_MANT_NUM-1] = (u8)(n >> 24);

   #if R_MANT_NUM > 1
	num->m[R_MANT_NUM-2] = (u8)(n >> 16);
   #endif

   #if R_MANT_NUM > 2
	num->m[R_MANT_NUM-3] = (u8)(n >> 8);
   #endif

   #if R_MANT_NUM > 3
	num->m[R_MANT_NUM-4] = (u8)n;
   #endif

  #else // R_BASE_BITS == R_MANT_LASTBITS

	// save shifted segments 8 bits
	RealSetLastSeg(num, (u8)(n >> (32 - R_MANT_LASTBITS)));

   #if R_MANT_NUM > 1
	num->m[R_MANT_NUM-2] = (u8)(n >> (24 - R_MANT_LASTBITS));
   #endif

   #if R_MANT_NUM > 2
	num->m[R_MANT_NUM-3] = (u8)(n >> (16 - R_MANT_LASTBITS));
   #endif

   #if R_MANT_NUM > 3
	num->m[R_MANT_NUM-4] = (u8)(n >> (8 - R_MANT_LASTBITS));
   #endif

   #if R_MANT_NUM > 4
	num->m[R_MANT_NUM-5] = (u8)(n << R_MANT_LASTBITS);
   #endif

  #endif // R_BASE_BITS == R_MANT_LASTBITS

 #else // R_BASE_BITS == 8

// segment is 16 bits

	// save whole segments 16 bit
  #if R_BASE_BITS == R_MANT_LASTBITS

	((u16*)num->m)[R_MANT_NUM-1] = (u16)(n >> 16);
   #if R_MANT_NUM > 1
	((u16*)num->m)[R_MANT_NUM-2] = (u16)n;
   #endif

  #else // R_BASE_BITS == R_MANT_LASTBITS

	// save shifted segments 16 bits
	RealSetLastSeg(num, (u16)(n >> (32 - R_MANT_LASTBITS)));

   #if R_MANT_NUM > 1
	((u16*)num->m)[R_MANT_NUM-2] = (u16)(n >> (16 - R_MANT_LASTBITS));
   #endif

   #if R_MANT_NUM > 2
	((u16*)num->m)[R_MANT_NUM-3] = (u16)(n << R_MANT_LASTBITS);
   #endif

  #endif // R_BASE_BITS == R_MANT_LASTBITS

 #endif // R_BASE_BITS == 8

	// save exponent
	RealSetExp(num, exp);

#endif // R_BASE_BYTES >= 4
}

// set signed integer value
void RealSetSInt(real* num, r_bases n)
{
	if (n < 0) // negative number
	{
		RealSetUInt(num, (r_base)-n);
		RealNeg(num);
	}
	else // positive number
		RealSetUInt(num, n);
}

void RealSetS8(real* num, s8 n)
{
	RealSetSInt(num, n);
}

void RealSetS16(real* num, s16 n)
{
#if R_BASE_BYTES >= 2 // size of segment is "word" or more
	RealSetSInt(num, n);
#else

	if (n < 0) // negative number
	{
		RealSetU16(num, (u16)-n);
		RealNeg(num);
	}
	else // positive number
		RealSetU16(num, n);
#endif
}

void RealSetS32(real* num, s32 n)
{
#if R_BASE_BYTES >= 4 // size of segment is "dword" or more
	RealSetSInt(num, n);
#else

	if (n < 0) // negative number
	{
		RealSetU32(num, (u32)-n);
		RealNeg(num);
	}
	else // positive number
		RealSetU32(num, n);
#endif
}

// get unsigned integer value (rounded towards zero)
r_base RealGetUInt(const real* num)
{
	r_base k;
	r_exp exp;
	Bool sign;

	// number is < 1 (negative or small positive), return zero
	sign = RealIsNeg(num);
	exp = RealGetExp(num);
	if (sign || (exp < R_EXP_BIAS)) return 0;

	// check overflow
	exp -= R_EXP_BIAS;
	if (exp >= R_BASE_BITS) return R_BASE_MASK;

#if R_BASE_BITS == R_MANT_LASTBITS // whole last segment

	k = ((r_base*)num->m)[R_MANT_NUM-1];

#else // R_BASE_BITS == R_MANT_LASTBITS

	// load higher part
	k = RealGetLastSeg(num);
	k <<= (R_BASE_BITS - R_MANT_LASTBITS);

 #if R_MANT_WHOLENUM > 0 // there is another valid segment
	k |= ((r_base*)num->m)[R_MANT_WHOLENUM-1] >> R_MANT_LASTBITS;
 #endif

#endif // R_BASE_BITS == R_MANT_LASTBITS

	// restore hidden bit
	k >>= 1;
	k |= R_BASE_LAST;

	// shift to valid position
	k >>= R_BASE_BITS - 1 - exp;

	return k;
}

u8 RealGetU8(const real* num)
{
	r_base n = RealGetUInt(num);
#if R_BASE_BYTES > 1 // size of segment is "word" or more
	if (n > 0xff) n = 0xff;
#endif
	return (u8)n;
}

u16 RealGetU16(const real* num)
{
#if R_BASE_BYTES > 1 // size of segment is "word" or more

	r_base n = RealGetUInt(num);
 #if R_BASE_BYTES > 2 // size of segment is "dword" or more
	if (n > 0xffff) n = 0xffff;
 #endif
	return (u16)n;

#else // R_BASE_BYTES > 1

// base is u8

	u16 k;
	r_exp exp;
	Bool sign;

	// number is < 1 (negative or small positive), return zero
	sign = RealIsNeg(num);
	exp = RealGetExp(num);
	if (sign || (exp < R_EXP_BIAS)) return 0;

	// check overflow
	exp -= R_EXP_BIAS;
	if (exp >= 16) return 0xffff;

 #if R_BASE_BITS == R_MANT_LASTBITS // whole last segment

	k = num->m[R_MANT_NUM-1];
	k <<= 8;
  #if R_MANT_NUM > 1
	k |= num->m[R_MANT_NUM-2];
  #endif

 #else // R_BASE_BITS == R_MANT_LASTBITS

	// load higher part
	k = RealGetLastSeg(num);
	k <<= 16 - R_MANT_LASTBITS;

  #if R_MANT_NUM > 1 // there is another valid segment
	k |= ((u16)num->m[R_MANT_NUM-2]) << (8 - R_MANT_LASTBITS);
  #endif

  #if R_MANT_NUM > 2
	k |= ((u16)num->m[R_MANT_NUM-3]) >> R_MANT_LASTBITS;
  #endif

 #endif // R_BASE_BITS == R_MANT_LASTBITS

	// restore hidden bit
	k >>= 1;
	k |= 0x8000;

	// shift to valid position
	k >>= 16 - 1 - exp;

	return k;

#endif // R_BASE_BYTES > 1

}

u32 RealGetU32(const real* num)
{
#if R_BASE_BYTES > 2 // size of segment is "dword" or more

	return RealGetUInt(num);

#elif R_BASE_BYTES > 1 // size of segment is "word"

// base is u16

	u32 k;
	r_exp exp;
	Bool sign;

	// number is < 1 (negative or small positive), return zero
	sign = RealIsNeg(num);
	exp = RealGetExp(num);
	if (sign || (exp < R_EXP_BIAS)) return 0;

	// check overflow
	exp -= R_EXP_BIAS;
	if (exp >= 32) return 0xffffffff;

 #if R_BASE_BITS == R_MANT_LASTBITS // whole last segment

	k = ((u16*)num->m)[R_MANT_NUM-1];
	k <<= 16;
  #if R_MANT_NUM > 1
	k |= ((u16*)num->m)[R_MANT_NUM-2];
  #endif

 #else // R_BASE_BITS == R_MANT_LASTBITS

	// load higher part
	k = RealGetLastSeg(num);
	k <<= (32 - R_MANT_LASTBITS);

  #if R_MANT_NUM > 1 // there is another valid segment
	k |= ((u32)(((u16*)num->m)[R_MANT_NUM-2])) << (16 - R_MANT_LASTBITS);
  #endif

  #if R_MANT_NUM > 2 // there is another valid segment
	k |= (((u16*)num->m)[R_MANT_NUM-3]) >> R_MANT_LASTBITS;
  #endif

 #endif // R_BASE_BITS == R_MANT_LASTBITS

	// restore hidden bit
	k >>= 1;
	k |= 0x80000000;

	// shift to valid position
	k >>= 32 - 1 - exp;

	return k;

#else // R_BASE_BYTES > 1 // size of segment is "byte"

// base is u8

	u32 k;
	r_exp exp;
	Bool sign;

	// number is < 1 (negative or small positive), return zero
	sign = RealIsNeg(num);
	exp = RealGetExp(num);
	if (sign || (exp < R_EXP_BIAS)) return 0;

	// check overflow
	exp -= R_EXP_BIAS;
	if (exp >= 32) return 0xffffffff;

 #if R_BASE_BITS == R_MANT_LASTBITS // whole last segment

	k = num->m[R_MANT_NUM-1];
	k <<= 24;

  #if R_MANT_NUM > 1
	k |= (u32)num->m[R_MANT_NUM-2] << 16;
  #endif

  #if R_MANT_NUM > 2
	k |= (u32)num->m[R_MANT_NUM-3] << 8;
  #endif

  #if R_MANT_NUM > 3
	k |= (u32)num->m[R_MANT_NUM-4];
  #endif

 #else // R_BASE_BITS == R_MANT_LASTBITS

	k = RealGetLastSeg(num);
	k <<= (32 - R_MANT_LASTBITS);

  #if R_MANT_NUM > 1
	k |= (u32)num->m[R_MANT_NUM-2] << (24 - R_MANT_LASTBITS);
  #endif

  #if R_MANT_NUM > 2
	k |= (u32)num->m[R_MANT_NUM-3] << (16 - R_MANT_LASTBITS);
  #endif

  #if R_MANT_NUM > 3
	k |= (u32)num->m[R_MANT_NUM-4] << (8 - R_MANT_LASTBITS);
  #endif

  #if R_MANT_NUM > 4
	k |= num->m[R_MANT_NUM-5] >> R_MANT_LASTBITS;
  #endif

 #endif // R_BASE_BITS == R_MANT_LASTBITS

	// restore hidden bit
	k >>= 1;
	k |= 0x80000000;

	// shift to valid position
	k >>= 32 - 1 - exp;

	return k;

#endif // R_BASE_BYTES > 1

}

// get signed integer value (rounded towards zero)
r_bases RealGetSInt(const real* num)
{
	r_base res;
	real tmp;

	if (RealIsNeg(num)) // negative number
	{
		RealCopy(&tmp, num);
		RealNeg(&tmp);
		res = RealGetUInt(&tmp);
		if (res > R_BASE_LAST) res = R_BASE_LAST;
		return -(r_bases)res;
	}

	// positive number
	res = RealGetUInt(num);
	if (res > (R_BASE_MASK>>1)) res = (R_BASE_MASK>>1);
	return (r_bases)res;
}

s8 RealGetS8(const real* num)
{
	r_bases n = RealGetSInt(num);
#if R_BASE_BYTES > 1 // size of segment is "word" or more
	if (n > 0x7f) n = 0x7f;
	if (n < (s8)-0x80) n = -0x80;
#endif
	return (s8)n;
}

s16 RealGetS16(const real* num)
{
#if R_BASE_BYTES > 1 // size of segment is "word" or more

	r_bases n = RealGetSInt(num);
#if R_BASE_BYTES > 2 // size of segment is "dword" or more
	if (n > 0x7fff) n = 0x7fff;
	if (n < (s16)-0x8000) n = -0x8000;
#endif
	return (s16)n;

#else // R_BASE_BYTES > 1

// base is 8 bit

	u16 res;
	real tmp;

	if (RealIsNeg(num))
	{
		RealCopy(&tmp, num);
		RealNeg(&tmp);
		res = RealGetU16(&tmp);
		if (res > 0x8000) res = 0x8000;
		return -(s16)res;
	}

	res = RealGetU16(num);
	if (res > 0x7fff) res = 0x7fff;
	return (s16)res;

#endif // R_BASE_BYTES > 1
}

s32 RealGetS32(const real* num)
{
#if R_BASE_BYTES > 2 // size of segment is "dword" or more

	return (s32)RealGetSInt(num);

#else // R_BASE_BYTES > 2

// base is 8 or 16 bit

	u32 res;
	real tmp;

	if (RealIsNeg(num))
	{
		RealCopy(&tmp, num);
		RealNeg(&tmp);
		res = RealGetU32(&tmp);
		if (res > 0x80000000) res = 0x80000000;
		return -(s32)res;
	}

	res = RealGetU32(num);
	if (res > 0x7fffffff) res = 0x7fffffff;
	return (s32)res;

#endif // R_BASE_BYTES > 2
}

// copy number
void RealCopy(real* dst, const real* src)
{
	int i;
	u8* d = dst->m;
	const u8* s = src->m;
	for (i = R_BYTES; i > 0; i--) *d++ = *s++;
}

// exchange numbers
void RealExch(real* num1, real* num2)
{
	int i;
	u8 k;
	u8* d = num1->m;
	u8* s = num2->m;
	for (i = R_BYTES; i > 0; i--)
	{
		k = *s;
		*s = *d;
		*d = k;
		s++;
		d++;
	}
}

// set constant from template in ROM
void RealConst(real* num, const real* temp)
{
	int i;
	u8* d = num->m;
	const u8* s = temp->m;
	for (i = R_BYTES; i > 0; i--)
	{
		*d++ = pgm_read_byte(s);
		s++;
	}
}

// export into memory number
void RealSave(const real* num, mreal* mem)
{
	u8 a;
	r_exp exp;
	s8 expH;
	int i, j, k;
	Bool sign;
	real m;
	r_base b, c;

	// copy number
	RealCopy(&m, num);

	// save sign
	sign = RealIsNeg(&m);

	// prepare exponent
	exp = RealGetExp(&m);
	expH = 0;
	if (exp == R_EXP_0)
	{
		exp = M_EXP_0;
	}
	else if (exp == R_EXP_INF)
	{
		exp = M_EXP_INF;
	}
	else
	{
		if (exp < R_EXP_BIAS) expH = -1;
		exp -= R_EXP_BIAS;
		exp += M_EXP_BIAS;
		if (exp < M_EXP_BIAS) expH++;
	}

	// clear exponent and sign
	RealSetExp(&m, 0);
	RealSetSign(&m, False);

	// round number
	i = R_MANT_BITS - M_MANT_BITS - 1; // rounding bit position
	if (i >= 0)
	{
		a = 1 << (i & 7); // bit mask
		i >>= 3; // byte index
		for (; i < R_MANT_BYTES; i++)
		{
			m.m[i] += a;
			if (m.m[i] >= a) break;
			a = 1;
		}

		// overflow, shift right
		if ((i == R_MANT_BYTES) || (RealGetExp(&m) != 0))
		{
			exp++;
			if (exp == 0) expH++;
			RealMantR(&m, 0);
			RealSetExp(&m, 0);
		}
	}
	
	// check overflow exponent
	if ((expH > 0) || ((expH == 0) && (exp > M_EXP_MAX)))
	{
		Error();
		RealSetInf(&m);
		exp = M_EXP_INF;
		expH = 0;
	}

	// check underflow exponent
	if ((expH < 0) || ((expH == 0) && (exp < M_EXP_MIN)))
	{
		RealSet0(&m);
		exp = M_EXP_0;
		expH = 0;
	}

	// prepare to copy mantissa
	i = R_MANT_NUM - 1; // last index of source mantissa
	j = M_MANT_NUM - 1; // last index of destination mantissa

	k = M_MANT_LASTBITS - R_MANT_LASTBITS; // shift segment
	if (k == 0) // no shift
	{
		b = RealGetLastSeg(&m);
		MRealSetLastSeg(mem, b);
		j--;
		i--;

		for (; j >= 0; j--)
		{
			b = m.m[i];
			mem->m[j] = b;
			i--;
		}
	}
	else if (k > 0) // shift mantissa up (left)
	{
		b = RealGetLastSeg(&m);
		i--;
		c = m.m[i];
		b <<= k;
		b |= (c >> (R_BASE_BITS - k));
		MRealSetLastSeg(mem, b);
		j--;

		for (; j >= 0; j--)
		{
			b = c;
			i--;
			c = m.m[i];
			b <<= k;
			b |= (c >> (R_BASE_BITS - k));
			mem->m[j] = b;
		}
	}
	else // shift mantissa down (right)
	{
		k = -k;
		b = RealGetLastSeg(&m);
		MRealSetLastSeg(mem, b >> k);
		c = b << (R_BASE_BITS - k);
		j--;
		i--;

		for (; j >= 0; j--)
		{
			b = m.m[i];
			i--;
			mem->m[j] = (b >> k) | c;
			c = b << (R_BASE_BITS - k);
		}
	}

	// set exponent
	MRealSetExp(mem, exp);

	// set sign
	mem->m[M_BYTES-1] &= 0x7f;
	if (sign) mem->m[M_BYTES-1] |= 0x80;
}

// import from memory number
void RealLoad(real* num, const mreal* mem)
{
	r_exp exp;
	s8 expH;
	int i, j, k;
	Bool sign;
	r_base b, c;

	// clear destination
	RealSet0(num);

	// save sign
	sign = (mem->m[M_BYTES-1] & 0x80) != 0;

	// prepare exponent
	exp = MRealGetExp(mem);
	expH = 0;
	if (exp == M_EXP_0)
	{
		return;
	}
	else if (exp == M_EXP_INF)
	{
		exp = R_EXP_INF;
	}
	else
	{
		if (exp < M_EXP_BIAS) expH = -1;
		exp -= M_EXP_BIAS;
		exp += R_EXP_BIAS;
		if (exp < R_EXP_BIAS) expH++;
	}

	// check overflow exponent
	if ((expH > 0) || ((expH == 0) && (exp > R_EXP_MAX)))
	{
		Error();
		RealSetInf(num);
		RealSetSign(num, sign);
		return;
	}

	// check underflow exponent
	if ((expH < 0) || ((expH == 0) && (exp < R_EXP_MIN)))
	{
		return;
	}

	// prepare to copy mantissa
	i = M_MANT_NUM - 1; // last index of source mantissa
	j = R_MANT_NUM - 1; // last index of destination mantissa

	k = R_MANT_LASTBITS - M_MANT_LASTBITS; // shift segment
	if (k == 0) // no shift
	{
		b = MRealGetLastSeg(mem);
		RealSetLastSeg(num, b);
		j--;
		i--;

		for (; i >= 0; i--)
		{
			b = mem->m[i];
			num->m[j] = b;
			j--;
		}
	}
	else if (k > 0) // shift mantissa up (left)
	{
		b = MRealGetLastSeg(mem);
		i--;
		c = mem->m[i];
		b <<= k;
		b |= (c >> (R_BASE_BITS - k));
		RealSetLastSeg(num, b);
		j--;

		for (; i >= 0; i--)
		{
			b = c;
			c = mem->m[i];
			b <<= k;
			b |= (c >> (R_BASE_BITS - k));
			num->m[j] = b;
			j--;
		}
		num->m[j] = (c << k);
	}
	else // shift mantissa down (right)
	{
		k = -k;
		b = MRealGetLastSeg(mem);
		RealSetLastSeg(num, b >> k);
		c = b << (R_BASE_BITS - k);
		j--;
		i--;

		for (; i >= 0; i--)
		{
			b = mem->m[i];
			num->m[j] = (b >> k) | c;
			j--;
			c = b << (R_BASE_BITS - k);
		}
		num->m[j] = c;
	}

	// set exponent
	RealSetExp(num, exp);

	// set sign
	RealSetSign(num, sign);
}

// ===========================================================================
//                          Arithmetics operations
// ===========================================================================

// add 2 numbers
void RealAdd(real* dst, const real* src1, const real* src2)
{
	Bool sign, sign1, sign2;
	r_exp exp, exp1, exp2;
	s8 carry, carry1, carry2;
	real s1, s2;
	int i;

	// 1st number is infinity 
	if (RealIsInf(src1))
	{
		if (RealIsInf(src2)) // 2nd number is infinity, too
		{
			if ((RealIsNeg(src1) && !RealIsNeg(src2)) ||
				(!RealIsNeg(src1) && RealIsNeg(src2))) // different signs
			{
				RealSet0(dst); // result is 0
			}
		}

		return; // output is 1st number
	}

	// 2nd number is infinity (but 1st one not) - output is 2nd number
	if (RealIsInf(src2))
	{
		RealCopy(dst, src2);
		return;
	}


	// 1st number is zero
	if (RealIsZero(src1))
	{
		RealCopy(dst, src2);
		return;
	}

	// 2nd number is zero
	if (RealIsZero(src2))
	{
		RealCopy(dst, src1);
		return;
	}

	// load operands, 2nd operand must be >= 1st one
	if (RealGetExp(src2) >= RealGetExp(src1))
	{
		RealCopy(&s1, src1);
		RealCopy(&s2, src2);
	}
	else
	{
		RealCopy(&s1, src2);
		RealCopy(&s2, src1);
	}

	// load exponents and signs
	exp1 = RealGetExp(&s1);
	exp2 = RealGetExp(&s2);
	sign1 = RealIsNeg(&s1);
	sign2 = RealIsNeg(&s2);

	// prepare 1st number
	carry1 = 1;
	if (sign1)
	{
		carry1 = ~carry1;
		if (RealMantNeg(&s1)) carry1++;
	}

	// prepare 2nd number
	carry2 = 1;
	if (sign2)
	{
		carry2 = ~carry2;
		if (RealMantNeg(&s2)) carry2++;
	}

	// prepare new exponent (= exponent of 2nd number)
	exp = exp2;

	// difference of exponents
	exp2 -= exp1;

	// difference is too high - only s2 left
	if (exp2 > R_MANT_BITS)
	{
		RealCopy(dst, &s2);
		RealSetSign(dst, sign2);
		RealSetExp(dst, exp);
		return;
	}

	// shift 1st number right to normalize numbers to the same exponent
	carry = 0;
	while (exp2 > 0)
	{
		carry = RealMantR(&s1, carry1 & 1);
		carry1 >>= 1;
		exp2--;
	}

	// add numbers
	carry = RealMantAdd(&s1, &s2, carry);
	carry1 = carry1 + carry2 + carry;

	// prepare sign flag and negate result
	sign = (carry1 >= 0) ? 0 : 1;
	if (sign)
	{
		carry1 = ~carry1;
		if (RealMantNeg(&s1)) carry1++;
	}

	// overflow
	if (carry1 > 1)
	{
		carry = RealMantR(&s1, carry1 & 1);
		carry1 >>= 1;
		exp++;
		if ((exp == 0) || (exp == R_EXP_INF))
		{
			RealSetInf(dst);
			RealSetSign(dst, sign);
			Error();
			return;
		}
		if (carry) RealMantInc(&s1); // result will not overflow
	}

	// normalize mantissa if result is too low
	for (i = R_MANT_BITS - R_TRIM; i > 0; i--)
	{
		if (carry1 != 0) break;
		carry1 = RealMantL(&s1, 0);
		exp--;
		if (exp == 0)
		{
			RealSet0(dst);
			return;
		}
	}

	// underflow, result is 0
	if ((i == 0) || (exp < R_EXP_MIN))
	{
		RealSet0(dst);
		return;
	}

	// set exponent
	RealSetExp(&s1, exp);

	// set sign
	RealSetSign(&s1, sign);

	// save result
	RealCopy(dst, &s1);
}


// subtract 2 numbers
void RealSub(real* dst, const real* src1, const real* src2)
{
	real m;
	RealCopy(&m, src2);
	RealNeg(&m);
	RealAdd(dst, src1, &m);
}

// bitwise operation (oper = OPER_AND, OPER_OR, OPER_XOR)
void RealBit(real* dst, const real* src1, const real* src2, u8 oper)
{
	Bool sign, sign1, sign2;
	r_exp exp, exp1, exp2;
	s8 carry, carry1, carry2;
	real s1, s2;
	int i;

	// load operands, 2nd operand must be >= 1st one
	if (RealGetExp(src2) >= RealGetExp(src1))
	{
		RealCopy(&s1, src1);
		RealCopy(&s2, src2);
	}
	else
	{
		RealCopy(&s1, src2);
		RealCopy(&s2, src1);
	}

	// load exponents and signs
	exp1 = RealGetExp(&s1);
	exp2 = RealGetExp(&s2);
	sign1 = RealIsNeg(&s1);
	sign2 = RealIsNeg(&s2);

	// prepare 1st number
	carry1 = 1;
	if (sign1)
	{
		carry1 = ~carry1;
		if (RealMantNeg(&s1)) carry1++;
	}

	// prepare 2nd number
	carry2 = 1;
	if (sign2)
	{
		carry2 = ~carry2;
		if (RealMantNeg(&s2)) carry2++;
	}

	// prepare new exponent (= exponent of 2nd number)
	exp = exp2;

	// difference of exponents
	exp2 -= exp1;
	if (exp2 > R_MANT_BITS) exp2 = R_MANT_BITS+1;

	// shift 1st number right to normalize numbers to the same exponent
	carry = 0;
	while (exp2 > 0)
	{
		carry = RealMantR(&s1, carry1 & 1);
		carry1 >>= 1;
		exp2--;
	}

	// bit operation
	if (oper == OPER_AND) // &
	{
		RealMantAnd(&s1, &s2);
		carry1 &= carry2;
	}
	else if (oper == OPER_OR) // |
	{
		RealMantOr(&s1, &s2);
		carry1 |= carry2;
	}
	else // if (oper == OPER_XOR) // ^
	{
		RealMantXor(&s1, &s2);
		carry1 ^= carry2;
	}

	// prepare sign flag and negate result
	sign = (carry1 >= 0) ? 0 : 1;
	if (sign)
	{
		carry1 = ~carry1;
		if (RealMantNeg(&s1)) carry1++;
	}

	// overflow
	if (carry1 > 1)
	{
		carry = RealMantR(&s1, carry1 & 1);
		carry1 >>= 1;
		exp++;
		if ((exp == 0) || (exp == R_EXP_INF))
		{
			RealSetInf(dst);
			RealSetSign(dst, sign);
			Error();
			return;
		}
		if (carry) RealMantInc(&s1); // result will not overflow
	}

	// normalize mantissa if result is too low
	for (i = R_MANT_BITS; i > 0; i--)
	{
		if (carry1 != 0) break;
		carry1 = RealMantL(&s1, 0);
		exp--;
		if (exp == 0)
		{
			RealSet0(dst);
			return;
		}
	}

	// underflow, result is 0
	if ((i == 0) || (exp < R_EXP_MIN))
	{
		RealSet0(dst);
		return;
	}

	// set exponent
	RealSetExp(&s1, exp);

	// set sign
	RealSetSign(&s1, sign);

	// save result
	RealCopy(dst, &s1);
}

// multiply 2 numbers
void RealMul(real* dst, const real* src1, const real* src2)
{
	Bool sign1, sign2;
	r_exp exp1, exp2;
	s8 expcarry, carry, carrydst;
	real s1, s2;
	int i;

	// save signs
	sign1 = RealIsNeg(src1);
	sign2 = RealIsNeg(src2);

	// 1st number is infinity
	if (RealIsInf(src1))
	{
		Error(); // overflow error
		if (RealIsZero(src2)) // 2nd number if zero
			RealSet1(dst); // result will be 1
		else
			RealSetInf(dst);
		RealSetSign(dst, sign1 ^ sign2);
		return;
	}

	// 2nd number is infinity
	if (RealIsInf(src2))
	{
		Error(); // overflow error
		if (RealIsZero(src1)) // 1st number is zero
			RealSet1(dst); // result will be 1
		else
			RealSetInf(dst);
		RealSetSign(dst, sign1 ^ sign2);
		return;
	}

	// some of numbers is zero - result will be zero
	if (RealIsZero(src1) || RealIsZero(src2))
	{
		RealSet0(dst);
		return;
	}

	// save exponents
	exp1 = RealGetExp(src1);
	exp2 = RealGetExp(src2);

	// prepare copy of src1 and src2
	RealCopy(&s1, src1);
	RealCopy(&s2, src2);

	// clear result register
	RealSet0(dst);
	carrydst = 0;

	// multiply mantissas
	carry = 1; // hidden bit of src1
	for (i = R_MANT_BITS+1; i > 0; i--)
	{
		// shift src1 right
		carry = RealMantR(&s1, carry);

		// add src2 to result
		if (carry)
		{
			carrydst += RealMantAdd(dst, &s2, 0);
			carrydst += 1; // add hidden bit of src2
		}

		// shift result right
		carry = RealMantR(dst, carrydst & 1);
		carrydst >>= 1;
	}

	// add exponents -> exp1, expcarry
	expcarry = 0;
	exp1 += exp2;
	if (exp1 < exp2) expcarry = 1;
	if (exp1 < R_EXP_BIAS-1) expcarry--;
	exp1 -= R_EXP_BIAS-1;

	// normalize mantissa if result is too low
	if (carrydst == 0)
	{
		carrydst = RealMantL(dst, 0);
		carry = RealMantL(&s1, 0);
		exp1--;
		if (exp1 == (r_exp)-1) expcarry--;
	}

	// round result
	if (carry)
	{
		if (RealMantInc(dst))
		{
			RealMantR(dst, 0);
			exp1++;
			if (exp1 == 0) expcarry++;
		}
	}

	// result is too small
	if ((exp1 < R_EXP_MIN) || (expcarry < 0))
	{
		RealSet0(dst);
		return;
	}

	// result is too high
	if ((exp1 > R_EXP_MAX) || (expcarry > 0))
	{
		RealSetInf(dst);
		exp1 = R_EXP_INF;
		Error();
	}

	// set result sign
	RealSetSign(dst, sign1 ^ sign2);

	// set exponent
	RealSetExp(dst, exp1);
}

// divide 2 numbers
void RealDiv(real* dst, const real* src1, const real* src2)
{
	Bool sign1, sign2;
	r_exp exp1, exp2;
	s8 expcarry, carry, carry1, carrydst, m;
	real s1, s2;
	int i;

	// save signs
	sign1 = RealIsNeg(src1);
	sign2 = RealIsNeg(src2);

	// divisor is 0, overflow
	if (RealIsZero(src2))
	{
		Error();
		if (RealIsZero(src1)) // dividend is zero too - result will be 1
			RealSet1(dst);
		else
		{
			RealSetInf(dst);
			if (sign1) RealNeg(dst);
		}
		return;
	}

	// dividend is zero - result will be zero
	if (RealIsZero(src1))
	{
		RealSet0(dst);
		return;
	}

	// dividend is infinity
	if (RealIsInf(src1))
	{
		Error();
		// divisor is infinity, too
		if (RealIsInf(src2))
			RealSet1(dst); // result will be 1
		else
			RealSetInf(dst); // result will be infinity
		RealSetSign(dst, sign1 ^ sign2);
		return;
	}


	// save exponents
	exp1 = RealGetExp(src1);
	exp2 = RealGetExp(src2);

	// prepare copy of src1 (dividend) and src2 (divisor)
	RealCopy(&s1, src1);
	RealCopy(&s2, src2);
	carry1 = 1; // hidden bit of src1

	// clear result register (quotient)
	RealSet0(dst);
	carrydst = 0;
	m = 0;

	// divide mantissas
	for (i = R_MANT_BITS+3; i > 0; i--)
	{
		// try to subtract divisor from dividend
		carry1 -= RealMantSub(&s1, &s2, 0);
		carry1 -= 1; // subtract hidden bit of src2
		carry = 1;

		// carry is set - dividend is smaler than divisor, restore dividend
		if (carry1 < 0)
		{
			carry1 += RealMantAdd(&s1, &s2, 0);
			carry1 += 1; // add hidden bit of src2
			carry = 0;
		}

		// add 2 last bits into temporary result
		if (i <= 2)
		{
			m <<= 1;
			m |= carry;
		}
		else
		{
			// add result bit to the accumulator
			carrydst <<= 1;
			carrydst |= RealMantL(dst, carry);
		}

		// shift dividend left
		carry1 <<= 1;
		carry1 |= RealMantL(&s1, 0);
	}

	// prepare exponent
	expcarry = 0;
	if (exp1 < exp2) expcarry = -1;
	exp1 -= exp2;
	exp1 += R_EXP_BIAS;
	if (exp1 < R_EXP_BIAS) expcarry++;

	// normalize mantissa if result is too low
	for (i = R_MANT_BITS; i > 0; i--)
	{
		if (carrydst != 0) break;
		carrydst = RealMantL(dst, ((m & 2) == 0) ? 0 : 1);
		m <<= 1;
		exp1--;
		if (exp1 == (r_exp)-1) expcarry--;
	}

	// round result
	if ((m & 2) != 0)
	{
		if (RealMantInc(dst))
		{
			RealMantR(dst, 0);
			exp1++;
			if (exp1 == 0) expcarry++;
		}
	}

	// result is too small
	if ((exp1 < R_EXP_MIN) || (expcarry < 0))
	{
		RealSet0(dst);
		return;
	}

	// result is too high
	if ((exp1 > R_EXP_MAX) || (expcarry > 0))
	{
		RealSetInf(dst);
		exp1 = R_EXP_INF;
		Error();
	}

	// set result sign
	RealSetSign(dst, sign1 ^ sign2);

	// set exponent
	RealSetExp(dst, exp1);
}

// power of 2 numbers (y = base^exp)
void RealPow(real* dst, const real* base, const real* exp)
{
	Bool sign;
	real m;

	// exponent is 0, result is 1
	if (RealIsZero(exp))
	{
		RealSet1(dst);
		return;
	}

	// base is 0
	if (RealIsZero(base))
	{
		// exponent is > 0, result is 0
		if (!RealIsNeg(exp))
		{
			RealSet0(dst);
			return;
		}

		// exponent is < 0, overflow error, divide by zero
		Error();
		RealSetInf(dst);
		return;
	}

	// negative base - allowed only if exponent is integer
	sign = False;
	if (RealIsNeg(base))
	{
		// truncate number to integer
		if (!RealTrunc(exp, &m, NULL)) // number is not integer
		{
			Error(); // incorrect parameter
		}
		else
		{
			// exp is integer; check if exponent is odd
			RealInc(&m, &m); // increment exponent
			RealDiv2(&m); // divide number by 2
			if (RealIsInt(&m)) sign = True; // exponent is odd, result will be negative
		}
	}

	// base is infinity
	if (RealIsInf(base))
	{
		// exponent is > 0
		if (!RealIsNeg(exp))
		{
			Error();
			RealSetInf(dst);
		}
		else
			RealSet0(dst);

		return;
	}

	// get logarithm of the base
	RealCopy(&m, exp); // save exponent (for case when dst = exp)
	RealCopy(dst, base);
	RealAbs(dst);
	RealLn(dst);

	// multiply by exponent
	RealMul(dst, dst, &m);

	// back to exponent
	RealExp(dst);

	// set sign
	RealSetSign(dst, sign);
}

// root of 2 numbers (y = Vbase^exp)
void RealRoot(real* dst, const real* base, const real* exp)
{
	real m;
	RealRec(&m, exp);
	RealPow(dst, base, &m);
}

// modulus (with floor rounding; result has same sign as divisor src2)
void RealModFloor(real* dst, const real* src1, const real* src2)
{
	// divide a/b
	real m;
	RealDiv(&m, src1, src2);

	// round down, get remainder
	RealFloor(&m, NULL, &m);

	// multiply remainder * divisor
	RealMul(dst, &m, src2);
}

// modulus (with trunc rounding; result has same sign as dividend src1)
void RealModTrunc(real* dst, const real* src1, const real* src2)
{
	// divide a/b
	real m;
	RealDiv(&m, src1, src2);

	// round towards zero, get remainder
	RealTrunc(&m, NULL, &m);

	// multiply remainder * divisor
	RealMul(dst, &m, src2);
}

// compare 2 numbers (returns -1: num1<num2, 0: num1=num2, +1: num1>num2)
s8 RealComp(const real* num1, const real* num2)
{
	real m;
	RealSub(&m, num1, num2);
	return RealComp0(&m);
}

// ===========================================================================
//                          Functions
// ===========================================================================

// compare number to zero (returns -1: num<0, 0: num=0, +1: num>0)
s8 RealComp0(const real* num)
{
	if (RealIsNeg(num)) return -1; // negative number
	if (RealIsZero(num)) return 0; // zero
	return 1; // positive
}

// square x^2
void RealSqr(real* dst, const real* src)
{
	RealMul(dst, src, src);
}

// reciprocal value 1/x
void RealRec(real* dst, const real* src)
{
	RealDiv(dst, &RealConst1, src);
}

// multiply by 2 (increases exponent by 1)
void RealMul2(real* num)
{
	Bool sign;
	r_exp exp = RealGetExp(num);

	if (exp == R_EXP_0) return; // number is zero
	if (exp >= R_EXP_MAX)
	{
		// overflow
		sign = RealIsNeg(num);
		Error();
		RealSetInf(num);
		RealSetSign(num, sign);
	}
	else
		RealSetExp(num, exp + 1);
}

// divide by 2 (decreases exponent by 1)
void RealDiv2(real* num)
{
	r_exp exp = RealGetExp(num);
	if (exp == R_EXP_INF) return; // number is infinity
	if (exp == R_EXP_0) return; // number is zero
	RealSetExp(num, exp - 1);
}

// shift left (multiply by power of 2)
void RealLeft(real* num, r_exp n)
{
	Bool sign;
	r_exp exp = RealGetExp(num);

	if ((exp == R_EXP_0) || (exp == R_EXP_INF)) return; // number is zero

	exp += n;
	if ((exp > R_EXP_MAX) || (exp < n))
	{
		// overflow
		sign = RealIsNeg(num);
		Error();
		RealSetInf(num);
		RealSetSign(num, sign);
	}
	else
		RealSetExp(num, exp);
}

// shift right (divide by power of 2)
void RealRight(real* num, r_exp n)
{
	r_exp exp = RealGetExp(num);
	if (exp == R_EXP_INF) return; // number is infinity
	if (exp == R_EXP_0) return; // number is zero

	if (exp <= n)
		RealSet0(num);
	else
		RealSetExp(num, exp - n);
}

// bitwise NOT
void RealNot(real* num)
{
	RealNeg(num);
	RealDec(num, num);
}

// increase number by 1
void RealInc(real* dst, const real* src)
{
	RealAdd(dst, src, &RealConst1);
}

// decrease number by 1
void RealDec(real* dst, const real* src)
{
	RealAdd(dst, src, &RealConstM1);
}

// rounding to nearest (returns sign of difference and rounding -1 up, 0 or +1 down)
//  num = source number
//  integ = output integer number, can be NULL
//  dif = difference num-integ = -0.5..+0.5, can be NULL
s8 RealRound(const real* num, real* integ, real* frac)
{
	real mint, mdif;

	// add rounding correction 0.5
	if (RealIsNeg(num))
		RealSub(&mint, num, &RealConstMore05); // subtract 0.5
	else
		RealAdd(&mint, num, &RealConstMore05); // add 0.5

	// truncate number - round number to nearest integer
	RealTruncPrec(&mint);

	// difference = number - integer, with trim lowest bits
	RealSub(&mdif, num, &mint);

	// output integer number
	if (integ != NULL) RealCopy(integ, &mint);

	// difference = 0, number was integer
	if (RealIsZero(&mdif))
	{
		// clear fractional part
		if (frac != NULL) RealSet0(frac);
		return 0;
	}

	// rounding
	if (frac != NULL) RealCopy(frac, &mdif);
	return (RealIsNeg(&mdif)) ? -1 : 1;		
}

// truncation towards zero (= "integer" function, returns True if integer)
//  num = source number
//  integ = output integer number, can be NULL
//  dif = difference num-integ = -1..+1, can be NULL
Bool RealTrunc(const real* num, real* integ, real* frac)
{
	s8 res;

	// negative flag
	Bool neg = RealIsNeg(num);

	// round number
	res = RealRound(num, integ, frac);

	// number was not negative
	if (!neg)
	{
		// correct rounding up
		if (res < 0)
		{
			if (integ != NULL) RealDec(integ, integ);
			if (frac != NULL) RealInc(frac, frac);
		}
	}
	else
	{
		// correct rounding down
		if (res > 0)
		{
			if (integ != NULL) RealInc(integ, integ);
			if (frac != NULL) RealDec(frac, frac);
		}
	}

	return (res == 0);
}

// rounding down (returns True if integer)
//  num = source number
//  integ = output integer number, can be NULL
//  dif = difference num-integ = 0..+1, can be NULL
Bool RealFloor(const real* num, real* integ, real* frac)
{
	// round number
	s8 res = RealRound(num, integ, frac);

	// correct rounding up
	if (res < 0)
	{
		if (integ != NULL) RealDec(integ, integ);
		if (frac != NULL) RealInc(frac, frac);
	}

	return (res == 0);
}

// rounding up (returns True if integer)
//  num = source number
//  integ = output integer number, can be NULL
//  dif = difference num-integ = -1..0, can be NULL
Bool RealCeil(const real* num, real* integ, real* frac)
{
	// round number
	s8 res = RealRound(num, integ, frac);

	// correct rounding down
	if (res > 0)
	{
		if (integ != NULL) RealInc(integ, integ);
		if (frac != NULL) RealDec(frac, frac);
	}

	return (res == 0);
}

// check if number is integer
Bool RealIsInt(const real* num)
{
	return RealRound(num, NULL, NULL) == 0;
}

// natural logarithm
//  Taylor serie D = (x-1)/(x+1),  xi+1 = xi*D^2*(i*2-1)/(i*2+1) ... x0 = 2*D
//  ln(x) = 2*D*(1 + D^2/3 + D^4/5 + D^6/7 ...)
void RealLn(real* num)
{
	r_exp exp;
	real x2, res, fac, tmp;
	Bool neg;
	s8 cmp;

	// check if number is <= 0 (invalid result)
	if (RealComp0(num) <= 0)
	{
		Error();
		if (RealIsZero(num))
		{
			RealSetMInf(num);
			return;
		}
		RealAbs(num);
	}

	// infinity
	if (RealIsInf(num))
	{
		Error();
		return;
	}

	// result is 0 if x=1
	cmp = RealComp(num, &RealConst1);
	if (cmp == 0)
	{
		RealSet0(num);
		return;
	}

	// do 1/x if <1 (negate result)
	neg = False;
	if (cmp < 0)
	{
		RealRec(num, num);
		neg = True;
		if (RealIsInf(num))
		{
			RealSetMInf(num);
			return;
		}
	}

	// load exponent
	exp = RealGetExp(num);

	// set exponent to 1-base, get range 1..1.999999
	RealSetExp(num, R_EXP_1);

	// prepare member D and result accumulator
	RealDec(&res, num); // res = x - 1
	RealInc(num, num); // num = x + 1
	RealDiv(num, &res, num); // num = (x - 1) / (x + 1) = D
	RealCopy(&res, num);

	// square D^2
	RealMul(&x2, num, num);

	// factorial coefficient -> 1
	RealSet1(&fac);

	// iterations
	for (;;)
	{
		// multiply member by D^2
		RealMul(num, num, &x2);

		// increase factorial coefficient by 2
		RealAdd(&fac, &fac, &RealConst2);

		// divide member by factorial coefficient
		RealDiv(&tmp, num, &fac);

		// add member to accumulator
		RealAdd(&res, &res, &tmp);

		// check epsilon of member
		if (((r_exps)(RealGetExp(&res) - RealGetExp(&tmp)) > (r_exps)(R_MANT_BITS + R_MANT_BITS/16)) ||
			RealIsZero(&tmp) ||
			RealIsInf(&tmp) ||
			RealIsInf(&res)) break;
	}

	// multiply by 2
	RealMul2(&res);

	// convert exponent to e-base and add it to the result
#if R_EXP_MAXBITS == 8 // save exponent
	RealSetU8(&fac, exp);
	RealSetU8(&x2, R_EXP_BIAS);
#elif R_EXP_MAXBITS == 16
	RealSetU16(&fac, exp);
	RealSetU16(&x2, R_EXP_BIAS);
#else // R_EXP_MAXBITS == 32
	RealSetU32(&fac, exp);
	RealSetU32(&x2, R_EXP_BIAS);
#endif
	RealSub(&fac, &fac, &x2);
	RealMul(&fac, &fac, &RealConstLn2);
	RealAdd(&res, &res, &fac);

	// negate result
	if (neg) RealNeg(&res);

	// copy result
	RealCopy(num, &res);
}

// ln(2) - calculates ln(2) constant
// - uses similar method as above, but does not require pre-calculated ln(2)
void RealLn2(real* num)
{
	real x2, res, fac, tmp;

	// prepare member D and result accumulator (x = 2, D = (2-1)/(2+1) = 1/3)
	RealSetU8(num, 3);
	RealRec(num, num);
	RealCopy(&res, num);

	// square D^2
	RealMul(&x2, num, num);

	// factorial coefficient -> 1
	RealSet1(&fac);

	// iterations
	for (;;)
	{
		// multiply member by D^2
		RealMul(num, num, &x2);

		// increase factorial coefficient by 2
		RealAdd(&fac, &fac, &RealConst2);

		// divide member by factorial coefficient
		RealDiv(&tmp, num, &fac);

		// add member to accumulator
		RealAdd(&res, &res, &tmp);

		// check epsilon of member
		if (((r_exps)(RealGetExp(&res) - RealGetExp(&tmp)) > (r_exps)(R_MANT_BITS + R_MANT_BITS/16)) ||
			RealIsInf(&tmp) ||
			RealIsInf(&res) ||
			RealIsZero(&tmp)) break;
	}

	// multiply by 2
	RealMul2(&res);

	// copy result
	RealCopy(num, &res);
}

// natural exponent
//  Taylor serie xi+1 = xi * x / (i+1) ... x0 = 1
//  exp(x) = 1 + x/1! + x^2/2! + x^3/3! + x^4/4! + x^5/5! + ...
void RealExp(real* num)
{
	real x, res, fac, big;
	r_expd expd;

	// zero -> 1
	if (RealIsZero(num))
	{
		RealSet1(num);
		return;
	}

	// check overflow
	if ((RealIsInf(num) && !RealIsNeg(num)) || (RealComp(num, &RealConstExpMax) > 0))
	{
		Error();
		RealSetInf(num);
		return;
	}

	// check underflow
	if ((RealIsInf(num) && RealIsNeg(num)) || (RealComp(num, &RealConstExpMin) < 0))
	{
		RealSet0(num);
		return;
	}

	// get integer part of exponent (integer /ln(2) integer *ln(2) subtract ... multiply)
	RealMul(&big, num, &RealConstRLn2); // convert to 2-base big = num / ln(2)

#if R_EXP_MAXBITS == 8 // load integer number exp = int(num / ln(2))
	expd = RealGetS16(&big);
#else // R_EXP_MAXBITS == 16 or 32
	expd = RealGetS32(&big);
#endif

#if R_EXP_MAXBITS == 8 // set exponent back
	RealSetS16(&big, expd);
#else // R_EXP_MAXBITS == 16 or 32
	RealSetS32(&big, expd);
#endif
	expd += R_EXP_BIAS;
	if (expd > R_EXP_MAX)
	{
		Error();
		RealSetInf(num);
		return;
	}

	if (expd < 0)
	{
		RealSet0(num);
		return;
	}

	RealMul(&big, &big, &RealConstLn2); // convert back to e-base big = int(num / ln(2)) * ln(2)
	RealSub(num, num, &big); // subtract integer exponent

	// prepare member
	RealSet1(&x);

	// factorial coefficient -> 1
	RealSet0(&fac);

	// result accumulator
	RealSet1(&res);

	// iterations
	for (;;)
	{
		// multiply member by x
		RealMul(&x, &x, num);

		// increase factorial coefficient
		RealInc(&fac, &fac);

		// divide member by factorial coefficient
		RealDiv(&x, &x, &fac);

		// add member to accumulator
		RealAdd(&res, &res, &x);

		// check epsilon of member
		if (((r_exps)(RealGetExp(&res) - RealGetExp(&x)) > (r_exps)(R_MANT_BITS + R_MANT_BITS/16)) ||
			RealIsInf(&x) ||
			RealIsInf(&res) ||
			RealIsZero(&x)) break;
	}

	// multiply result by integer part and save result
	RealSet1(&big);
	RealSetExp(&big, (r_exp)expd);
	RealMul(num, &res, &big);
}

// decimal logarithm
void RealLog10(real* num)
{
	RealLn(num);
	RealMul(num, num, &RealConstRLn10);
}

// decimal exponent
void RealExp10(real* num)
{
	RealMul(num, num, &RealConstLn10);
	RealExp(num);
}

// binary logarithm
void RealLog2(real* num)
{
	RealLn(num);
	RealMul(num, num, &RealConstRLn2);
}

// binary exponent
void RealExp2(real* num)
{
	RealMul(num, num, &RealConstLn2);
	RealExp(num);
}

// sqrt
void RealSqrt(real* num)
{
	// base is 0, result is 0
	if (RealIsZero(num))
	{
		RealSet0(num);
		return;
	}

	// negative base - error
	if (RealIsNeg(num))
	{
		Error(); // incorrect parameter
		RealAbs(num);
	}

	// get logarithm of the base
	RealLn(num);

	// divide logarithm / 2
	RealDiv2(num);

	// back to exponent
	RealExp(num);
}

// convert angle from current angle unit to radians
void RealToRad(real* num)
{
	if (Unit == UNIT_RAD) return;
	if (Unit == UNIT_DEG)
		RealMul(num, num, &RealConstPi180);
	else
		RealMul(num, num, &RealConstPi200);
}

// convert angle from radians to current angle unit
void RealFromRad(real* num)
{
	if (Unit == UNIT_RAD) return;
	if (Unit == UNIT_DEG)
		RealMul(num, num, &RealConst180Pi);
	else
		RealMul(num, num, &RealConst200Pi);
}

// normalize angle in radians into range 0..PI*2 (= 0..360)
void RealNormRad(real* num)
{
	RealModFloor(num, num, &RealConstPi2);
}

// sine
//  Taylor serie xi+1 = xi * -x^2/((2*i)*(2*i+1)) ... x0 = x
//  sin(x) = x/1! - x^3/3! + x^5/5! - x^7/7! + x^9/9! - ...
void RealSin(real* num)
{
	Bool neg;
	real x2, res, fac;

	// normalize angle into range 0..PI*2
	RealNormRad(num);

	// normalize angle into range 0..PI
	neg = False;
	if (RealComp(num, &RealConstPi) >= 0) // if angle >= PI
	{
		RealSub(num, &RealConstPi2, num); // correction angle = PI*2 - angle
		neg = True; // negate result
	}

	// number is zero, result will be zero
	if ((RealComp(num, &RealConst0) == 0) ||
		(RealComp(num, &RealConstPi) == 0))
	{
		RealSet0(num);
		return;
	}

	// number is Pi/2, result will be 1
	if (RealComp(num, &RealConstPi05) == 0)
	{
		RealSet1(num);
		RealSetSign(num, neg);
		return;
	}

	// square -x^2
	RealMul(&x2, num, num);
	RealNeg(&x2);

	// prepare result accumulator -> x
	RealCopy(&res, num);

	// prepare factorial coefficient -> 1
	RealSet1(&fac);

	// iterations
	for (;;)
	{
		// multiply member by -x^2
		RealMul(num, num, &x2);

		// increment factorial coefficient
		RealInc(&fac, &fac);

		// divide member by factorial coefficient
		RealDiv(num, num, &fac);

		// increment factorial coefficient
		RealInc(&fac, &fac);

		// divide member by factorial coefficient
		RealDiv(num, num, &fac);

		// add member to accumulator
		RealAdd(&res, &res, num);

		// check epsilon of member
		if (((r_exps)(RealGetExp(&res) - RealGetExp(num)) > (r_exps)(R_MANT_BITS + R_MANT_BITS/16)) ||
			RealIsInf(num) ||
			RealIsInf(&res) ||
			RealIsZero(num)) break;
	}

	// limit range to -1..+1
	if (RealGetExp(&res) >= R_EXP_1) RealSet1(&res);

	// negate result
	if (neg) RealNeg(&res);

	// copy result
	RealCopy(num, &res);
}

// cosine
//  cos = sin(x+PI/2)
void RealCos(real* num)
{
	RealAdd(num, num, &RealConstPi05);
	RealSin(num);
}

// tangent
//  tan = sin/cos
void RealTan(real* num)
{
	real m;
	RealCopy(&m, num);
	RealSin(&m);
	RealCos(num);
	RealDiv(num, &m, num);
}

// cotangent
//  cotan = cos/sin
void RealCoTan(real* num)
{
	real m;
	RealCopy(&m, num);
	RealSin(&m);
	RealCos(num);
	RealDiv(num, num, &m);
}

// arcus sine (result is in range -PI/2..+PI/2)
//  Taylor serie xi+1 = xi * (2*i-1)^2*x^2 / ((2*i)*(2*i+1)) ... x0 = x
//  arcsin(x) = x + x^3/2/3 + 3*x^5/2/4/5 + 3*5*x^7/2/4/6/7 +
//  For x > 0.75 we use alternate method via arccos: sqrt(1 - x^2)
void RealASin(real* num)
{
	Bool neg, alt;
	real x2, res, fac;
	s8 cmp;

	// number is zero, result will be zero
	if (RealIsZero(num)) return;

	// absolute value
	neg = RealIsNeg(num);
	RealAbs(num);

	// if number is >= 1, use pre-calculated value PI/2 (error if > 1)
	cmp = RealComp(num, &RealConst1);
	if (cmp >= 0)
	{
		if (cmp > 0) Error();
		RealCopy(num, &RealConstPi05);
		if (neg) RealNeg(num);
		return;
	}

	// check if use alternate method (if > 0.75, we use sqrt(1 - x^2))
	alt = (RealComp(num, &RealConst075) > 0);
	if (alt)
	{
		RealSqr(num, num); // x^2
		RealSub(num, &RealConst1, num); // 1 - x^2
		RealSqrt(num); // sqrt(1 - x^2)
	}

	// square x^2
	RealMul(&x2, num, num);

	// prepare result accumulator -> x
	RealCopy(&res, num);

	// prepare factorial coefficient -> 1
	RealSet1(&fac);

	// iterations
	for (;;)
	{
		// multiply member by x^2
		RealMul(num, num, &x2);

		// multiply member by coefficient^2
		RealMul(num, num, &fac);
		RealMul(num, num, &fac);

		// increment factorial coefficient
		RealInc(&fac, &fac);

		// divide member by factorial coefficient
		RealDiv(num, num, &fac);

		// increment factorial coefficient
		RealInc(&fac, &fac);

		// divide member by factorial coefficient
		RealDiv(num, num, &fac);

		// add member to accumulator
		RealAdd(&res, &res, num);

		// check epsilon of member
		if (((r_exps)(RealGetExp(&res) - RealGetExp(num)) > (r_exps)(R_MANT_BITS + R_MANT_BITS/16)) || 
			RealIsInf(num) ||
			RealIsInf(&res) ||
			RealIsZero(num)) break;
	}

	// correction for alternate method
	if (alt)
	{
		RealNeg(&res);
		RealAdd(&res, &RealConstPi05, &res);
	}

	// negate result
	if (neg) RealNeg(&res);

	// copy result
	RealCopy(num, &res);
}

// arcus cosine (result is in range 0..PI)
void RealACos(real* num)
{
	RealASin(num);
	RealNeg(num);
	RealAdd(num, num, &RealConstPi05);
}

// arcus tangent (result is in range -PI/2..+PI/2)
void RealATan(real* num)
{
	real m;
	RealMul(&m, num, num); // x^2
	RealInc(&m, &m); // 1 + x^2
	RealSqrt(&m); // sqrt(1 + x^2)
	RealDiv(num, num, &m); // x / sqrt(1 + x^2)
	RealASin(num); // atan(x) = asin(x / sqrt(1 + x^2))
}

// arcus cotangent (result is in range -PI/2..+PI/2)
void RealACoTan(real* num)
{
	RealATan(num);
	RealSub(num, &RealConstPi05, num);
}

// hyperbolic sine
//  sinh(x) = (e^x - e^-x)/2
void RealSinH(real* num)
{
	real m;
	RealExp(num);
	RealDiv(&m, &RealConst1, num);
	RealSub(num, num, &m);
	RealDiv2(num);
}

// hyperbolic cosine
//  cosh(x) = (e^x + e^-x)/2
void RealCosH(real* num)
{
	real m;
	RealExp(num);
	RealDiv(&m, &RealConst1, num);
	RealAdd(num, num, &m);
	RealDiv2(num);
}

// hyperbolic tangent
//  tanh(x) = (e^2x - 1) / (e^2x + 1)
void RealTanH(real* num)
{
	real m;
	RealExp(num);
	RealSqr(num, num);
	RealCopy(&m, num);
	RealDec(num, num);
	RealInc(&m, &m);
	RealDiv(num, num, &m);
}

// hyperbolic cotangent
//  cotanh(x) = (e^2x + 1) / (e^2x - 1)
void RealCoTanH(real* num)
{
	real m;
	RealExp(num);
	RealSqr(num, num);
	RealCopy(&m, num);
	RealInc(num, num);
	RealDec(&m, &m);
	RealDiv(num, num, &m);
}

// hyperbolic secans
//  sech(x) = 2*e^x / (e^2x + 1)
void RealSecH(real* num)
{
	real m;
	RealExp(num);
	RealCopy(&m, num);
	RealSqr(&m, &m);
	RealInc(&m, &m);
	RealMul2(num);
	RealDiv(num, num, &m);
}

// hyperbolic cosecant
//  csch(x) = 2*e^x / (e^2x - 1)
void RealCscH(real* num)
{
	real m;
	RealExp(num);
	RealCopy(&m, num);
	RealSqr(&m, &m);
	RealDec(&m, &m);
	RealMul2(num);
	RealDiv(num, num, &m);
}

// areasine, inverse hyperbolic sine
//  arsh(x) = ln(x + sqrt(x^2 + 1))
void RealArSinH(real* num)
{
	real m;
	RealMul(&m, num, num);
	RealInc(&m, &m);
	RealSqrt(&m);
	RealAdd(num, num, &m);
	RealLn(num);
}

// areacosine, inverse hyperbolic cosine
//  arch(x) = ln(x + sqrt(x^2 - 1))
void RealArCosH(real* num)
{
	real m;
	RealMul(&m, num, num);
	RealDec(&m, &m);
	RealSqrt(&m);
	RealAdd(num, num, &m);
	RealLn(num);
}

// areatangent, inverse hyperbolic tangent
//  arth(x) = ln((1 + x)/(1 - x))/2
void RealArTanH(real* num)
{
	real m;
	RealSub(&m, &RealConst1, num);
	RealInc(num, num);
	RealDiv(num, num, &m);
	RealLn(num);
	RealDiv2(num);
}

// areacotangent, inverse hyperbolic cotangent
//  arcth(x) = ln((x + 1)/(x - 1))/2
void RealArCoTanH(real* num)
{
	real m;
	RealSub(&m, num, &RealConst1);
	RealInc(num, num);
	RealDiv(num, num, &m);
	RealLn(num);
	RealDiv2(num);
}

// areasecant, inverse hyperbolic secant
//  arsch(x) = ln((1 + sqrt(1 - x^2))/x)
void RealArSecH(real* num)
{
	real m;
	RealMul(&m, num, num);
	RealSub(&m, &RealConst1, &m);
	RealSqrt(&m);
	RealInc(&m, &m);
	RealDiv(num, &m, num);
	RealLn(num);
}

// areacosecant, inverse hyperbolic cosecant
//  arcsch(x) = ln((1 - sqrt(1 + x^2))/x) ... x<0
//  arcsch(x) = ln((1 + sqrt(1 + x^2))/x) ... x>0
void RealArCscH(real* num)
{
	real m;
	RealMul(&m, num, num);
	RealInc(&m, &m);
	RealSqrt(&m);
	if (RealIsNeg(num))
		RealSub(&m, &RealConst1, &m);
	else
		RealAdd(&m, &RealConst1, &m);
	RealDiv(num, &m, num);
	RealLn(num);
}

// set seed of random generator
void SetRndSeed(u32 seed)
{
	RandSeed = seed;
}	

// get seed of random generator
u32 GetRndSeed()
{
	return RandSeed;
}

// shift random generator
u32 RndShift()
{
	u32 rnd = RandSeed * 214013 + 2531011;
	RandSeed = rnd;
	return rnd;
}

// set random number in range 0<= .. <1 (32-bit precision)
void RealSetRnd(real* num)
{
	r_exp exp;
	RealSetU32(num, RndShift());
	exp = RealGetExp(num);
	if (exp >= 32)
		exp -= 32;
	else
		exp = 0;
	RealSetExp(num, exp);
}

// random number in range 0<= .. <num (32-bit precision)
void RealRnd(real* num)
{
	real m;
	RealSetRnd(&m);
	RealMul(num, num, &m);
}

// integer factorial
void RealFactInt(real* num, u16 n)
{
	u16 i;
	real m;
	RealSet1(num);

	for (i = 1; i <= n; i++)
	{
		RealSetU16(&m, i);
		if (RealGetExp(num) == R_EXP_MAX)
		{
			RealMul(num, num, &m);
			break; // overflow
		}
		RealMul(num, num, &m);
	}
}

// non-integer factorial approximation (precission 27 digits or more)
void RealFact(real* num)
{
// Stieltjes, Gamma function, continued fraction:
// p = a0/(x + a1/(x + a2/(x + ... an/x
//		a0 = 1 / 12
//		a1 = 1 / 30
//		a2 = 53 / 210
//		a3 = 195 / 371
//		a4 = 22999 / 22737
//		a5 = 29944523 / 19733142
//		a6 = 109535241009 / 48264275462
//		a7 = 29404527905795295658 / 9769214287853155785
//		a8 = 455377030420113432210116914702 / 113084128923675014537885725485
//		a9 = 26370812569397719001931992945645578779849 / 5271244267917980801966553649147604697542
//		a10 = 152537496709054809881638897472985990866753853122697839 / 24274291553105128438297398108902195365373879212227726
//		a11 = 100043420063777451042472529806266909090824649341814868347109676190691 / 13346384670164266280033479022693768890138348905413621178450736182873
// q = p + ln(2*pi)/2 - x + (x + 1/2)*ln(x)
// y = exp(p)
// precision digits, without work-around correction: 1: 6, 2: 10, 3: 13, 4: 15, 5: 17, 6: 19, 7: 20, 8: 22, 9: 23, 10: 24, 11: 25, 12: 26, 13..14: 27, 15: 28, 100: 48, 1000: 73
// precision digits with work-around correction: 1..4: 27, 5: 28, 6..8: 29, 9: 30, 10..11: 31, 12..13: 32, 14..17: 33, 18..19: 34, 100: 49, 1000: 74

	int i;
	real p, q, c;
	Bool corr;
	Bool integer;

#if R_EXP_BITS < 12
#define FACTCORR (R_EXP_BITS-1)
#else
#define FACTCORR 12
#endif

	// number is integer
	integer = RealIsInt(num);

	// work-around to raise precission of low numbers - shift calculations to higher values
	corr = RealGetExp(num) < R_EXP_1 + FACTCORR;
	if (corr)
	{
		RealSet1(&q);
		i = (int)(R_EXP_1 + FACTCORR - RealGetExp(num));
		for (; i > 0; i--)
		{
			RealInc(num, num);
			RealMul(&q, &q, num);
		}
	}

	RealCopy(&p, num);

	for (i = FACT_COEFF-1; i > 0; i--)
	{
		RealDiv(&p, &RealConstFactA[i], &p); // ai/xi
		RealAdd(&p, &p, num); // x + ai/xi
	}

	RealDiv(&p, &RealConstFactA[0], &p); // a0/x0

	RealAdd(&p, &p, &RealConstLnPi22); // p + ln(2*pi)/2
	RealSub(&p, &p, num); // p + ln(2*pi)/2 - x

	RealCopy(&c, num);
	RealLn(&c); // ln(x)
	RealAdd(num, num, &RealConst05); // x + 0.5
	RealMul(num, num, &c); // ln(x)*(x + 0.5)

	RealAdd(num, num, &p);
	RealExp(num);

	// work-around correction
	if (corr) RealDiv(num, num, &q);

	if (integer) RealRound(num, num, NULL);
}

// natural logarithm of factorial (using approximation)
void RealFactLn(real* num)
{
	int i;
	real p, q, c;
	Bool corr;

	// work-around to raise precission of low numbers - shift calculations to higher values
	corr = RealGetExp(num) < R_EXP_1 + FACTCORR;
	if (corr)
	{
		RealSet1(&q);
		i = (int)(R_EXP_1 + FACTCORR - RealGetExp(num));
		for (; i > 0; i--)
		{
			RealInc(num, num);
			RealMul(&q, &q, num);
		}
		RealLn(&q); // convert to logarithm
	}

	RealCopy(&p, num);

	for (i = FACT_COEFF-1; i > 0; i--)
	{
		RealDiv(&p, &RealConstFactA[i], &p); // ai/xi
		RealAdd(&p, &p, num); // x + ai/xi
	}

	RealDiv(&p, &RealConstFactA[0], &p); // a0/x0

	RealAdd(&p, &p, &RealConstLnPi22); // p + ln(2*pi)/2
	RealSub(&p, &p, num); // p + ln(2*pi)/2 - x

	RealCopy(&c, num);
	RealLn(&c); // ln(x)
	RealAdd(num, num, &RealConst05); // x + 0.5
	RealMul(num, num, &c); // ln(x)*(x + 0.5)

	RealAdd(num, num, &p);

	// work-around correction
	if (corr) RealSub(num, num, &q);
}

// decimal logarithm of factorial (using approximation)
void RealFactLog(real* num)
{
	RealFactLn(num);
	RealDiv(num, num, &RealConstLn10);
}

// check precision of factorial approximation - get number of valid digits of RealFactSmooth
//  edd(n) = -log10(abs(1 - approximation(n) / n!))
//  http://www.luschny.de/math/factorial/approx/SimpleCases.html

		// test factorial
/*		int i;
		real a, b, c;
		s16 x;

		for (i = 1; i < 1000; i++)
		{
			RealFactInt(&a, (u16)i);
			RealToEdit(&a);

			RealSetU16(&b, (u16)i);
			RealFact(&b);
			RealToEdit(&b);

			RealFactCheck(&c, (u16)i);
			x = RealGetS16(&c);

// here set breakpoint and watch x, i and EditBuf:
			x = x;
		}
*/

/*
// to use - comment correction "if (integer) RealRound(num);"
void RealFactCheck(real* num, u16 n)
{
	real n1, n2;

	// integer factorial
	RealFactInt(&n1, n);

	// factorial approximation
	RealSetU16(&n2, n);
	RealFact(&n2);

	// compare result
	RealDiv(&n2, &n2, &n1); // fact(n)/n!

	RealSet1(num);
	RealSub(num, num, &n2); // 1 - fact(n)/n!
	RealAbs(num); // abs(1 - fact(n)/n!)

	if (RealIsZero(num)) // result are equal
	{
		RealSetU16(num, R_MANT_DIG);
		return;
	}

	RealLog10(num); // log10(abs(1 - fact(n)/n!))
	RealRound(num); // round number of digits
	RealNeg(num);
}

*/
