
// ****************************************************************************
//                                 
//                              Decode number
//
// ****************************************************************************

#include "include.h"

u8 Base;		// numeric radix base

// text formating flags
Bool EEMode; // EE mode with exponent; not set together with Eng
Bool EngMode; // engineering technical mode, exponent is multiply of 3
s8 Fix; // fixed decimal places (0.. digits, or FIX_OFF=off)

// ponter to text for RealFromText function
const char* FromText;
Bool FromRom;

// split number to mantissa and exponent, returns decimal exponent
r_exps SplitNum(real* num)
{
	real m, m2;
	Bool expneg;
	s8 inx, inc;
	r_exp exp;
	r_exps exps;

	// get current binary exponent
	exp = RealGetExp(num);

	// store exponent
#if R_EXP_MAXBITS == 8
	RealSetU8(&m, exp);
	RealSetU8(&m2, R_EXP_BIAS);
#elif R_EXP_MAXBITS == 16
	RealSetU16(&m, exp);
	RealSetU16(&m2, R_EXP_BIAS);
#else // R_EXP_MAXBITS == 32
	RealSetU32(&m, exp);
	RealSetU32(&m2, R_EXP_BIAS);
#endif

	// subtract bias
	RealNeg(&m2);
	RealAdd(&m, &m, &m2);

	// correction: negative exponent - 1, positive exponent + 2
	if (exp < R_EXP_BIAS)
		RealAdd(&m, &m, &RealConstM1); // m = exp2 - 1
	else
		RealAdd(&m, &m, &RealConst2); // m = exp2 + 2

	// convert binary exponent to decimal - multiply by log10(2)
	if (Base == BASE_DEC) // decimal base
		RealMul(&m, &m, &RealConstLog2);
	else if (Base == BASE_HEX) // hexadecimal base
	{
		RealDiv2(&m);
		RealDiv2(&m);
	}
	else if (Base == BASE_OCT) // octal radix
	{
		RealSetU8(&m2, 3);
		RealDiv(&m, &m, &m2);
	}

	// get exponent sign
	expneg = RealIsNeg(&m);
	RealAbs(&m);

	// load decimal exponent
#if R_EXP_MAXBITS == 8
	exp = RealGetU8(&m);
#elif R_EXP_MAXBITS == 16
	exp = RealGetU16(&m);
#else // R_EXP_MAXBITS == 32
	exp = RealGetU32(&m);
#endif

	// save output signed exponent
	exps = (expneg) ? -(r_exps)exp : exp;

	// non-decimal exponent
	if (Base != BASE_DEC)
	{
		if (Base == BASE_HEX)
		{
			RealSetExp(num, RealGetExp(num) - exps*4);
			while (RealGetExp(num) < R_EXP_1)
			{
				RealMul2(num);
				RealMul2(num);
				RealMul2(num);
				RealMul2(num);
				exps--;
			}

			while (RealGetExp(num) >= R_EXP_1+4)
			{
				RealDiv2(num);
				RealDiv2(num);
				RealDiv2(num);
				RealDiv2(num);
				exps++;
			}
		}
		else if (Base == BASE_OCT)
		{
			RealSetExp(num, RealGetExp(num) - exps*3);
			while (RealGetExp(num) < R_EXP_1)
			{
				RealMul2(num);
				RealMul2(num);
				RealMul2(num);
				exps--;
			}
			while (RealGetExp(num) >= R_EXP_1+3)
			{
				RealDiv2(num);
				RealDiv2(num);
				RealDiv2(num);
				exps++;
			}
		}
		else
		{
			RealSetExp(num, RealGetExp(num) - exps);
			while (RealGetExp(num) < R_EXP_1)
			{
				RealMul2(num);
				exps--;
			}
			while (RealGetExp(num) >= R_EXP_1+1)
			{
				RealDiv2(num);
				exps++;
			}
		}
		return exps;
	}

	// divide mantissa by decimal exponent
	inx = R_DECEXP; // index of '1' (middle of the table)
	inc = (expneg) ? 1 : -1; // increment of the index

	while (exp != 0)
	{
		inx += inc; // shift index
		if (exp & 1) RealMul(num, num, &RealConstExp[inx]);
		exp >>= 1;
	}

	// mantissa is now in range 0.25..5 (or 0), correction to get into range 1..9.9999
	if (RealGetExp(num) < R_EXP_1)
	{
		RealMul(num, num, &RealConst10);
		exps--;
	}
	return exps;
}

// convert number into edit buffer
void RealToEdit(const real* num)
{
	Bool restart; // restarting code (after overflow 9.9999 -> 10.0000)
	real m; // mantissa 1..1.9999
	r_exps exps; // decimal exponent signed
	r_exp exp, exp2; // decimal exponent unsigned (absolute value)
	u8 totnum; // total digits of mantissa (without sign and without decimal point)
	u8 expnum; // number of exponent digits (including sign)
	u8 zeronum; // number of additional zeroes on start of number (between decimal point and valid digits)
	u8 intnum; // digits of integer part
	u8 decnum; // digits of decimal part
	u8 n;
	Bool err; // overflow error
	u8 olderr = ErrCode;

	real tmp, tmp2;
	char* dst;
	char* src;

	// copy number
	RealCopy(&m, num);
	err = (RealGetExp(&m) == R_EXP_INF);

	// clear edit buffer
	EditBufClear();
	exps = 0;

	// check negative number, store sign
	if (RealIsNeg(&m))
	{
		RealAbs(&m);
		EditBuf[0] = '-';
	}
	else
		EditBuf[0] = ' ';

	// check if number is not 0
	if (!RealIsZero(&m))
	{
		// split number to mantissa and exponent
		exps = SplitNum(&m);
	}

	// restart conversion if rounding increases exponent (increase mantissa 9.9999 -> 10.0000)
	restart = False;

// ---- here we have positive mantissa (m) in range 1..9.9999 and signed exponent (exp)

	// prepare engineering mode (cannot repeat during restart)
	intnum = 1; // digits of integer part
	if (EngMode)
	{
		RealSetS32(&tmp, exps); // load exponent
		if ((Base == BASE_DEC) || (Base == BASE_OCT))
			RealSetU8(&tmp2, 3); // load 3
		else
			RealSetU8(&tmp2, 4); // load 4
		RealModFloor(&tmp, &tmp, &tmp2); // operation modulo, rounding floor (we may have negative exponent)
		intnum = RealGetU8(&tmp); // load integer digits-1
		exps -= intnum; // shift exponent to be multiply of 3
		intnum++; // digits of integer part
	}

Restart:

	// prepare flags
	totnum = MAX_DIG; // total number of digits of mantissa (without sign and without decimal point)
	expnum = 0; // number of exponent digits (including sign)
	zeronum = 0; // number of additional zeroes on start of number (between decimal point and valid digits)

	// check if use exponent mode
	exp = (exps < 0) ? -exps : exps;
	if (EEMode || EngMode || (exps >= MAX_DIG) || ((exps < -MAX_DIG) && (Fix == FIX_OFF)))
	{
		// decode exponent to end of buffer
		dst = &EditBuf[BUF_SIZE-1]; // end of buffer
		do {
			exp2 = exp / 10;
			if (err)
				*dst = '9';
			else
				*dst = (char)(exp - exp2*10 + '0');
			exp = exp2;
			expnum++;
			dst--;
		} while (exp != 0);

		// store sign
		*dst = (exps < 0) ? '-' : '+';
		expnum++;

		// decrease length of mantissa
		totnum -= expnum;

		// number of decimal digits
		decnum = totnum - intnum;

		// limit by fix
		if (Fix != FIX_OFF)
		{
			if (Fix < decnum) decnum = Fix;
		}
	}

	// prepare normal mode
	else
	{
		// integer digits = exponent + 1
		if (exps >= 0)
			intnum = (u8)exps + 1;
		else
		{
			intnum = 0; // no integer digit (may be expanded to '0.' later)
			if (exps >= -MAX_DIG-1)
				zeronum = (u8)(-exps - 1); // zero digits
			else
				zeronum = MAX_DIG+1;
		}

		// decimal digits
		if (intnum + zeronum <= totnum)
			decnum = totnum - intnum - zeronum;
		else
		{
			decnum = 0;
			zeronum = totnum - intnum;
		}

		// limit by fix
		if (Fix != FIX_OFF)
		{
			if (Fix < zeronum) zeronum = Fix;
			if (Fix < zeronum + decnum) decnum = Fix - zeronum;
		}
	}

	// save number of digits of exponent
	ExpDig = expnum;

	// limit digits by display length
	n = BUF_SIZE - 2 - expnum; // buffer size without sign, decimal point and exponent
	if (n < intnum) intnum = n; // limit integers
	n -= intnum;
	if (n < zeronum) zeronum = n; // limit zero digits
	n -= zeronum;
	if (n < decnum) decnum = n; // limit decimal digits

	// total number of valid digits of mantissa
	totnum = intnum + decnum;

	// rounding - only if valid digits and if not restarting
	if ((totnum > 0) && (totnum <= R_MAXROUND) && !restart)
	{
		if (Base == BASE_DEC) // decimal base
		{
			// add rounding correction to the mantissa
			RealAdd(&m, &m, &RealConstRound[totnum-1]);

			// raise mantissa 9.9999 -> 10.0000, normalize and restart
			if (RealComp(&m, &RealConst10) >= 0)
			{
				RealMul(&m, &m, &RealConst01);
				exps++;
				restart = True;
				if (EngMode)
				{
					if (intnum > 3)
					{
						intnum -= 3;
						exps += 3;
					}
				}
				goto Restart;
			}
		}
		else if (Base == BASE_HEX)
		{
			RealSet0(&tmp);
			RealSetExp(&tmp, R_EXP_1 - 1 - (totnum-1)*4);
			RealAdd(&m, &m, &tmp);

			// raise mantissa 9.9999 -> 10.0000, normalize and restart
			if (RealGetExp(&m) >= R_EXP_1 + 4)
			{
				RealRight(&m, 4);
				exps++;
				restart = True;
				if (EngMode)
				{
					if (intnum > 4)
					{
						intnum -= 4;
						exps += 4;
					}
				}
				goto Restart;
			}
		}
		else if (Base == BASE_OCT)
		{
			RealSet0(&tmp);
			RealSetExp(&tmp, R_EXP_1 - 1 - (totnum-1)*3);
			RealAdd(&m, &m, &tmp);

			// raise mantissa 9.9999 -> 10.0000, normalize and restart
			if (RealGetExp(&m) >= R_EXP_1 + 3)
			{
				RealRight(&m, 3);
				exps++;
				restart = True;
				if (EngMode)
				{
					if (intnum > 3)
					{
						intnum -= 3;
						exps += 3;
					}
				}
				goto Restart;
			}
		}
		else // if (Base == BASE_BIN)
		{
			RealSet0(&tmp);
			RealSetExp(&tmp, R_EXP_1 - 1 - (totnum-1));
			RealAdd(&m, &m, &tmp);

			// raise mantissa 9.9999 -> 10.0000, normalize and restart
			if (RealGetExp(&m) >= R_EXP_1 + 1)
			{
				RealDiv2(&m);
				exps++;
				restart = True;
				if (EngMode)
				{
					if (intnum > 4)
					{
						intnum -= 4;
						exps += 4;
					}
				}
				goto Restart;
			}
		}
	}

	// start decimal point if integer digits = 0
	PointOn = True;
	dst = &EditBuf[1];
	if (intnum == 0) *dst++ = '.';

	// store initial zeroes
	for (; zeronum > 0; zeronum--) *dst++ = '0';

	// decode digits of mantissa
	for (; totnum > 0; totnum--)
	{
		// get integer part of the number
		n = RealGetU8(&m);
		RealSetU8(&tmp, n);

		if (n > 9) n += 'A' - '9' - 1;

		// store digit
		if (err)
			*dst++ = '9';
		else
			*dst++ = n + '0';

		// subtract integer part
		RealNeg(&tmp);
		RealAdd(&m, &m, &tmp);

		// multiply mantissa by 10
		if (Base == BASE_DEC)
			RealMul(&m, &m, &RealConst10);
		else if (Base == BASE_OCT)
		{
			RealMul2(&m);
			RealMul2(&m);
			RealMul2(&m);
		}
		else if (Base == BASE_HEX)
		{
			RealMul2(&m);
			RealMul2(&m);
			RealMul2(&m);
			RealMul2(&m);
		}
		else
			RealMul2(&m);

		// store decimal point
		intnum--;
		if (intnum == 0) *dst++ = '.';
	}

	// delete ending zeroes (only if not fixed mode)
	if (Fix == FIX_OFF)
	{
		for (; decnum > 0; decnum--)
		{
			if (dst[-1] != '0') break;
			dst--;
			*dst = ' ';
		}

		// delete single decimal point
		if (dst[-1] == '.')
		{
			dst--;
			*dst = ' ';
			PointOn = False;
		}
	}

	// length of mantissa
	totnum = (u8)(dst - EditBuf);
	EditDig = totnum; // save length of mantissa

	// shift mantissa right
	expnum = BUF_SIZE - expnum; // index of start of exponent

	if (expnum > totnum)
	{
		// move mantissa
		dst = &EditBuf[expnum];
		src = &EditBuf[totnum];
		for (; totnum > 0; totnum--)
		{
			dst--;
			src--;
			*dst = *src;
		}

		// insert first '0'
		if ((dst[1] == '.') || (dst[1] == ' '))
		{
			dst--;
			*dst = dst[1]; // sign
			dst[1] = '0';
			EditDig++;
		}

		// clear rest of buffer
		totnum = (u8)(dst - EditBuf);
		for (; totnum > 0; totnum--)
		{
			dst--;
			*dst = ' ';
		}
	}

	// ignore error when decoding infinity
	ErrCode = olderr;
}

// load one character (0=end of text)
char FromChar()
{
	char ch;
	if (FromRom)
		ch = pgm_read_byte(FromText);
	else
		ch = *FromText;
	if (ch == 0) return 0;
	FromText++;
	return ch;
}

// load number from ASCIIZ text
void RealFromText(real* num, const char* text, Bool rom)
{
	real mul; // decimal multiplier 0.1, 0.01, ...
	real tmp;
	char sign; // sign
	char ch;
	u8 inx, inc;
	r_exp exp;

	// save pointer
	FromText = text;
	FromRom = rom;

	// clear number
	RealSet0(num);

	// prepare decimal multiplier
	RealSet1(&mul);

	// skip spaces and load 1st character
	do {
		ch = FromChar();
	} while ((ch == ' ') || (ch == '+'));

	// load sign of the mantissa
	sign = False;
	if (ch == '-')
	{
		sign = True;
		ch = FromChar();
	}

	// add integer digits to the mantissa
	for (;;)
	{
		// decimal point
		if (ch == '.')
		{
			ch = FromChar();
			break;
		}

		// end of mantissa and multiply accumulator
		if (Base == BASE_DEC)
		{
			if ((ch < '0') || (ch > '9')) break;
			ch -= '0';
			RealMul(num, num, &RealConst10);
		}
		else if (Base == BASE_HEX)
		{
			if ((ch < '0') || ((ch > '9') && (ch < 'A')) || (ch > 'F')) break;
			ch -= '0';
			if (ch > 9) ch -= 'A' - '9' - 1;
			RealMul2(num);
			RealMul2(num);
			RealMul2(num);
			RealMul2(num);
		}
		else if (Base == BASE_OCT)
		{
			if ((ch < '0') || (ch > '7')) break;
			ch -= '0';
			RealMul2(num);
			RealMul2(num);
			RealMul2(num);
		}
		else // if (Base == BASE_BIN)
		{
			if ((ch < '0') || (ch > '1')) break;
			ch -= '0';
			RealMul2(num);
		}

		// add new digit to the accumulator
		RealSetU8(&tmp, ch);
		RealAdd(num, num, &tmp);

		// load next character
		ch = FromChar();
	}

	// add decimal digits to the mantissa
	for (;;)
	{
		// end of mantissa and shift multiplier
		if (Base == BASE_DEC)
		{
			if ((ch < '0') || (ch > '9')) break;
			ch -= '0';
			RealMul(&mul, &mul, &RealConst01);
		}
		else if (Base == BASE_HEX)
		{
			if ((ch < '0') || ((ch > '9') && (ch < 'A')) || (ch > 'F')) break;
			ch -= '0';
			if (ch > 9) ch -= 'A' - '9' - 1;
			RealDiv2(&mul);
			RealDiv2(&mul);
			RealDiv2(&mul);
			RealDiv2(&mul);
		}
		else if (Base == BASE_OCT)
		{
			if ((ch < '0') || (ch > '7')) break;
			ch -= '0';
			RealDiv2(&mul);
			RealDiv2(&mul);
			RealDiv2(&mul);
		}
		else // if (Base == BASE_BIN)
		{
			if ((ch < '0') || (ch > '1')) break;
			ch -= '0';
			RealDiv2(&mul);
		}

		// multiply digit by multiplier and add to the accumulator
		RealSetU8(&tmp, ch);
		RealMul(&tmp, &tmp, &mul);
		RealAdd(num, num, &tmp);

		// load next character
		ch = FromChar();
	}

	// set sign of the mantissa
	if (sign) RealNeg(num);

	// sign of exponent
	sign = False;
	while ((ch == 'e') || (ch == 'E') || (ch == '+')) ch = FromChar();
	if (ch == '-')
	{
		sign = True;
		ch = FromChar();
	}
	
	// load digits of exponent
	exp = 0;
	for (;;)
	{
		if ((ch < '0') || (ch > '9')) break;
		exp = exp*10 + ch - '0';
		ch = FromChar();
	}

	// multiply mantissa by exponent
	inx = R_DECEXP; // index of '1' (middle of the table)
	inc = (sign) ? -1 : 1; // increment of the index
	while (exp != 0)
	{
		inx += inc; // shift index
		if (exp & 1) RealMul(num, num, &RealConstExp[inx]);
		exp >>= 1;
	}
}
