// Generate constants for TI-59

#define FILENAME "const.c"

// ***************************************************************************
//
//                             Main module
//
// ***************************************************************************

#include "include.h"
#include <stdio.h>

#pragma warning( disable : 4101) // unused variable
#pragma warning(disable : 4996) // unsafe function

FILE* File;

u8 pgm_read_byte(const u8* s) { return *s; }

// error code (ERR_*)
u8 ErrCode;

// soft error, can continue
jmp_buf exception_env;
void Error()
{
	if (ErrCode != ERR_FAULT) ErrCode = ERR_ERROR;
}

// fatal error, cannot continue
void Fatal()
{
	ErrCode = ERR_FAULT;
	longjmp(exception_env, 1);
}

// Edit buffer (terminated with 0)
char EditBuf[BUF_SIZE+1];

// parameters of edit buffer
Bool EditOn; // edit mode is on (entering number)
Bool PointOn; // decimal point was entered
Bool ExpOn; // entering exponent
Bool XValid; // register X is valid
u8 EditDig; // number of digits of mantissa in edit mode (including decimal point and sign)
u8 ExpDig; // number of digits of exponent in edit mode (including sign)

// clear text buffer (set to value '0')
void EditBufClear()
{
	int i;

	PointOn = False; // decimal point not entered
	EditDig = 2; // mantissa is 2 digits
	ExpDig = 0; // no exponent digits

	for (i = 0; i < BUF_SIZE-1; i++) EditBuf[i] = ' ';
	EditBuf[i] = '0';
	EditBuf[i+1] = 0;
}

// output one number
void Const1(real* num)
{
	mreal m;
	int i;

	RealSave(num, &m);

	fprintf(File, "{{");

	for (i = 0; i < M_BYTES-1; i++)
		fprintf(File, "0x%02X,", m.m[i]);

	fprintf(File, "0x%02X", m.m[i]);

	fprintf(File, "}}");
}

// output one constant
void Const(real* num, const char* name, const char* comment)
{
	fprintf(File, "const real %s PROGMEM = ", name);

	Const1(num);

	fprintf(File, "; // %s\n", comment);
}

int main(int argc, char* argv[])
{
	int i, i2, i3;
	real r;
	mreal m;

	// exception handling
	if (setjmp(exception_env))
	{

	}

	// first initialize
	else
	{
		// initialize random generator (@TODO: use EEPROM)
		SetRndSeed(1234);

		Unit = UNIT_DEG;

		Fix = FIX_OFF;
		//EEMode = True;
		//EngMode = True;

		// initialize real numbers
		RealInit();

		printf("Generating " FILENAME "...");

		// open output file
		File = fopen(FILENAME, "wb");
		if (File == NULL)
		{
			printf("\nCannot create output file " FILENAME "\n");
			return 2;
		}

		fprintf(File,
				"// ***************************************************************************\n"
				"//\n"
				"//                            Pre-generated Constants\n"
				"//\n"
				"// ***************************************************************************\n"
				"// Auto-generated, do not modify!\n"
				"// %d bytes, exponent %d bits, mantissa %d bits\n\n"
				"#include \"include.h\"\n\n",
				M_BYTES, M_EXP_BITS, M_MANT_BITS);

		fprintf(File, "// templates of constants in ROM\n");
		Const(&RealConstLn2, "RealTempLn2", "ln(2)");
		Const(&RealConstRLn2, "RealTempRLn2", "1/ln(2)");
		Const(&RealConstLn10, "RealTempLn10", "ln(10)");
		Const(&RealConstRLn10, "RealTempRLn10", "1/ln(10)");
		Const(&RealConstLog2, "RealTempLog2", "log(2)");
		MRealSetMax(&m); RealLoad(&r, &m); RealLn(&r); Const(&r, "RealTempExpMax", "max. exp(x)");
		MRealSetMin(&m); RealLoad(&r, &m); RealLn(&r); Const(&r, "RealTempExpMin", "min. exp(x)");
		Const(&RealConstEul, "RealTempEul", "Eul");
		Const(&RealConstPi05, "RealTempPi05", "pi/2");
		Const(&RealConstPi, "RealTempPi", "pi");
		Const(&RealConstPi2, "RealTempPi2", "pi*2");
		Const(&RealConstLnPi22, "RealTempLnPi22", "ln(pi*2)/2");
		Const(&RealConst180Pi, "RealTemp180Pi", "180/pi");
		Const(&RealConstPi180, "RealTempPi180", "pi/180");
		Const(&RealConst200Pi, "RealTemp200Pi", "200/pi");
		Const(&RealConstPi200, "RealTempPi200", "pi/200");

		// decimal exponents
		fprintf(File, "\n// decimal exponents (index R_DECEXP = number '1')\n"
			"const real RealTempExp[R_DECEXP*2+1] PROGMEM = {\n");
		for (i = 0; i < M_DECEXP*2+1; i++)
		{
			fprintf(File, "\t");
			Const1(&RealConstExp[i+R_DECEXP-M_DECEXP]);
			i2 = 0;
			if (i < M_DECEXP)
			{
				i2 = -1;
				i3 = M_DECEXP-1;
				while (i3 > i)
				{
					i2 *= 2;
					i3--;
				}		
			}
			else if (i > M_DECEXP)
			{
				i2 = 1;
				i3 = M_DECEXP+1;
				while (i3 < i)
				{
					i2 *= 2;
					i3++;
				}
			}
			fprintf(File, ",\t// 1e%d\n", i2);
		}
		fprintf(File, "};\n\n");

		// rounding correction
		fprintf(File, "// rounding corrections (0.5, 0.05,...)\n"
				"const real RealTempRound[R_MAXROUND] PROGMEM = {\n");

		for (i = 0; i < M_MANT_DIG+2; i++)
		{
			fprintf(File, "\t");
			Const1(&RealConstRound[i]);
			fprintf(File, ",\t// 5e%d\n", -(i + 1));
		}
		fprintf(File, "};\n\n");

		// factorial coefficients
		fprintf(File, "// factorial coefficients\n"
				"const real RealTempFactA[FACT_COEFF] PROGMEM = {\n");

		for (i = 0; i < FACT_COEFF; i++)
		{
			fprintf(File, "\t");
			Const1(&RealConstFactA[i]);
			fprintf(File, ",\t// a%d\n", i);
		}
		fprintf(File, "};\n\n");

		// close output file
		fclose(File);

		printf("\n");
	}

	return 0;
}

