
// ****************************************************************************
//
//                      Drawing to canvas at 2-bit format
//
// ****************************************************************************
// PicoLibSDK - Alternative SDK library for Raspberry Pico and RP2040
// Copyright (c) 2023 Miroslav Nemecek, Panda38@seznam.cz, hardyplotter2@gmail.com
// 	https://github.com/Panda381/PicoLibSDK
//	https://www.breatharian.eu/hw/picolibsdk/index_en.html
//	https://github.com/pajenicko/picopad
//	https://picopad.eu/en/
// License:
//	This source code is freely available for any purpose, including commercial.
//	It is possible to take and modify the code or parts of it, without restriction.

#if USE_DRAWCAN && USE_DRAWCAN3		// 1=use drawing canvas library (lib_drawcan*.c, lib_drawcan*.h)

#ifndef _LIB_DRAWCAN3_H
#define _LIB_DRAWCAN3_H

#ifdef __cplusplus
extern "C" {
#endif

// default drawing canvas for 3-bit format
extern sDrawCan DrawCan3;

// current drawing canvas for 3-bit format
extern sDrawCan* pDrawCan3;

// set default drawing canvas for 3-bit format
void SetDrawCan3(sDrawCan* can);

// get current width, height and buffer of the drawing canvas of 3-bit format
INLINE int Draw3Width() { return pDrawCan3->w; }
INLINE int Draw3Height() { return pDrawCan3->h; }
INLINE u8* Draw3Buf() { return pDrawCan3->buf; }

// convert RGB888 color to 3-bit pixel color RGB111
u16 Draw3ColRgb(u8 r, u8 g, u8 b);

#if USE_RAND		// use Random number generator (lib_rand.c, lib_rand.h)
// random 3-bit pixel color RGB111
u16 Draw3ColRand();
#endif

// 3-bit colors, format RGB111 (components are 0..255)
#define COLOR3(r,g,b) ((u8)( (((r)>>5)&4) | (((g)>>6)&2) | (((b)>>7)&1) ))
// - base colors
#define COL3_BLACK	COLOR3(  0,  0,  0)
#define COL3_BLUE	COLOR3(  0,  0,255)
#define COL3_GREEN	COLOR3(  0,255,  0)
#define COL3_CYAN	COLOR3(  0,255,255)
#define COL3_RED	COLOR3(255,  0,  0)
#define COL3_MAGENTA	COLOR3(255,  0,255)
#define COL3_YELLOW	COLOR3(255,255,  0)
#define COL3_GRAY	COLOR3(255,255,255)
#define COL3_WHITE	COLOR3(255,255,255)
#define COL3_RANDOM	(RandU8()&7)

// calculate pitch of graphics line from image width (get length of line in bytes, rounded up to 4-byte word)
int Draw3Pitch(int w);

// calculate max. width of image from the pitch (pitch is length of line in bytes, rounded up to 4-byte word)
int Draw3MaxWidth(int pitch);

// ----- Clear canvas

// Clear canvas with color
void DrawCan3ClearCol(sDrawCan* can, u16 col);
void Draw3ClearCol(u16 col);

// Clear canvas with black color
void DrawCan3Clear(sDrawCan* can);
void Draw3Clear();

// ----- Draw point

// Draw point (does not check clipping limits)
void DrawCan3Point_Fast(sDrawCan* can, int x, int y, u16 col);
void Draw3Point_Fast(int x, int y, u16 col);

// Draw point (checks clipping limits)
void DrawCan3Point(sDrawCan* can, int x, int y, u16 col);
void Draw3Point(int x, int y, u16 col);

// Draw point inverted (does not check clipping limits)
void DrawCan3PointInv_Fast(sDrawCan* can, int x, int y);
void Draw3PointInv_Fast(int x, int y);

// Draw point inverted (checks clipping limits)
void DrawCan3PointInv(sDrawCan* can, int x, int y);
void Draw3PointInv(int x, int y);

// ----- Get point

// Get point (does not check clipping limits)
u16 DrawCan3GetPoint_Fast(sDrawCan* can, int x, int y);
u16 Draw3GetPoint_Fast(int x, int y);

// Get point (checks clipping limits)
u16 DrawCan3GetPoint(sDrawCan* can, int x, int y);
u16 Draw3GetPoint(int x, int y);

// ----- Draw rectangle

// Draw rectangle (does not check clipping limits; dimensions must be > 0)
void DrawCan3Rect_Fast(sDrawCan* can, int x, int y, int w, int h, u16 col);
void Draw3Rect_Fast(int x, int y, int w, int h, u16 col);

// Draw rectangle (checks clipping limits; negative dimensions flip rectangle)
void DrawCan3Rect(sDrawCan* can, int x, int y, int w, int h, u16 col);
void Draw3Rect(int x, int y, int w, int h, u16 col);

// Draw rectangle inverted (does not check clipping limits; dimensions must be > 0)
void DrawCan3RectInv_Fast(sDrawCan* can, int x, int y, int w, int h);
void Draw3RectInv_Fast(int x, int y, int w, int h);

// Draw rectangle inverted (checks clipping limits; negative dimensions flip rectangle)
void DrawCan3RectInv(sDrawCan* can, int x, int y, int w, int h);
void Draw3RectInv(int x, int y, int w, int h);

// ----- Draw horizontal line

// Draw horizontal line (does not check clipping limits; width must be > 0)
void DrawCan3HLine_Fast(sDrawCan* can, int x, int y, int w, u16 col);
void Draw3HLine_Fast(int x, int y, int w, u16 col);

// Draw horizontal line (checks clipping limits; negative width flip line)
void DrawCan3HLine(sDrawCan* can, int x, int y, int w, u16 col);
void Draw3HLine(int x, int y, int w, u16 col);

// Draw horizontal line inverted (does not check clipping limits; width must be > 0)
void DrawCan3HLineInv_Fast(sDrawCan* can, int x, int y, int w);
void Draw3HLineInv_Fast(int x, int y, int w);

// Draw horizontal line inverted (checks clipping limits; negative width flip line)
void DrawCan3HLineInv(sDrawCan* can, int x, int y, int w);
void Draw3HLineInv(int x, int y, int w);

// ----- Draw vertical line

// Draw vertical line (does not check clipping limits; height must be > 0)
void DrawCan3VLine_Fast(sDrawCan* can, int x, int y, int h, u16 col);
void Draw3VLine_Fast(int x, int y, int h, u16 col);

// Draw vertical line (checks clipping limits; negative height flip line)
void DrawCan3VLine(sDrawCan* can, int x, int y, int h, u16 col);
void Draw3VLine(int x, int y, int h, u16 col);

// Draw vertical line inverted (does not check clipping limits; height must be > 0)
void DrawCan3VLineInv_Fast(sDrawCan* can, int x, int y, int h);
void Draw3VLineInv_Fast(int x, int y, int h);

// Draw vertical line inverted (checks clipping limits; negative height flip line)
void DrawCan3VLineInv(sDrawCan* can, int x, int y, int h);
void Draw3VLineInv(int x, int y, int h);

// ----- Draw frame

// Draw 1-pixel frame (does not check clipping limits; dimensions must be > 0)
void DrawCan3Frame_Fast(sDrawCan* can, int x, int y, int w, int h, u16 col_light, u16 col_dark);
void Draw3Frame_Fast(int x, int y, int w, int h, u16 col_light, u16 col_dark);

// Draw 1-pixel frame (checks clipping limits; negative dimensions flip frame)
void DrawCan3Frame(sDrawCan* can, int x, int y, int w, int h, u16 col_light, u16 col_dark);
void Draw3Frame(int x, int y, int w, int h, u16 col_light, u16 col_dark);

// Draw 1-pixel frame inverted (does not check clipping limits; dimensions must be > 0)
void DrawCan3FrameInv_Fast(sDrawCan* can, int x, int y, int w, int h);
void Draw3FrameInv_Fast(int x, int y, int w, int h);

// Draw 1-pixel frame inverted (checks clipping limits; negative dimensions flip frame)
void DrawCan3FrameInv(sDrawCan* can, int x, int y, int w, int h);
void Draw3FrameInv(int x, int y, int w, int h);

// Draw thick frame (does not check clipping limits; dimensions and thick must be > 0)
void DrawCan3FrameW_Fast(sDrawCan* can, int x, int y, int w, int h, u16 col, int thick);
void Draw3FrameW_Fast(int x, int y, int w, int h, u16 col, int thick);

// Draw thick frame (checks clipping limits; negative dimensions flip frame)
void DrawCan3FrameW(sDrawCan* can, int x, int y, int w, int h, u16 col, int thick);
void Draw3FrameW(int x, int y, int w, int h, u16 col, int thick);

// Draw thick frame inverted (does not check clipping limits; dimensions and thick must be > 0)
void DrawCan3FrameWInv_Fast(sDrawCan* can, int x, int y, int w, int h, int thick);
void Draw3FrameWInv_Fast(int x, int y, int w, int h, int thick);

// Draw thick frame inverted (checks clipping limits; negative dimensions flip frame)
void DrawCan3FrameWInv(sDrawCan* can, int x, int y, int w, int h, int thick);
void Draw3FrameWInv(int x, int y, int w, int h, int thick);

// ----- Draw line

// Draw line with overlapped pixels (does not check clipping limits; function used internally by the "draw thick line" function)
//  over ... draw overlapped pixels
void DrawCan3LineOver_Fast(sDrawCan* can, int x1, int y1, int x2, int y2, u16 col, Bool over);
void Draw3LineOver_Fast(int x1, int y1, int x2, int y2, u16 col, Bool over);

// Draw line (does not check clipping limits)
void DrawCan3Line_Fast(sDrawCan* can, int x1, int y1, int x2, int y2, u16 col);
void Draw3Line_Fast(int x1, int y1, int x2, int y2, u16 col);

// Draw line with overlapped pixels (checks clipping limits; function used internally by the "draw thick line" function)
void DrawCan3LineOver(sDrawCan* can, int x1, int y1, int x2, int y2, u16 col, Bool over);
void Draw3LineOver(int x1, int y1, int x2, int y2, u16 col, Bool over);

// Draw line (checks clipping limits)
void DrawCan3Line(sDrawCan* can, int x1, int y1, int x2, int y2, u16 col);
void Draw3Line(int x1, int y1, int x2, int y2, u16 col);

// Draw line inverted with overlapped pixels (does not check clipping limits; function used internally by the "draw thick line" function)
void DrawCan3LineOverInv_Fast(sDrawCan* can, int x1, int y1, int x2, int y2, Bool over);
void Draw3LineOverInv_Fast(int x1, int y1, int x2, int y2, Bool over);

// Draw line inverted (does not check clipping limits)
void DrawCan3LineInv_Fast(sDrawCan* can, int x1, int y1, int x2, int y2);
void Draw3LineInv_Fast(int x1, int y1, int x2, int y2);

// Draw line inverted with overlapped pixels (checks clipping limits; function used internally by the "draw thick line" function)
void DrawCan3LineOverInv(sDrawCan* can, int x1, int y1, int x2, int y2, Bool over);
void Draw3LineOverInv(int x1, int y1, int x2, int y2, Bool over);

// Draw line inverted (checks clipping limits)
void DrawCan3LineInv(sDrawCan* can, int x1, int y1, int x2, int y2);
void Draw3LineInv(int x1, int y1, int x2, int y2);

// Draw thick line (does not check clipping limits)
//  thick ... thick of line in pixels
//  round ... draw round ends
void DrawCan3LineW_Fast(sDrawCan* can, int x1, int y1, int x2, int y2, u16 col, int thick, Bool round);
void Draw3LineW_Fast(int x1, int y1, int x2, int y2, u16 col, int thick, Bool round);

// Draw thick line (checks clipping limits)
//  thick ... thick of line in pixels
//  round ... draw round ends
void DrawCan3LineW(sDrawCan* can, int x1, int y1, int x2, int y2, u16 col, int thick, Bool round);
void Draw3LineW(int x1, int y1, int x2, int y2, u16 col, int thick, Bool round);

// Draw thick line inverted (does not check clipping limits)
//  thick ... thick of line in pixels
//  round ... draw round ends
void DrawCan3LineWInv_Fast(sDrawCan* can, int x1, int y1, int x2, int y2, int thick);
void Draw3LineWInv_Fast(int x1, int y1, int x2, int y2, int thick);

// Draw thick line inverted (checks clipping limits)
//  thick ... thick of line in pixels
//  round ... draw round ends
void DrawCan3LineWInv(sDrawCan* can, int x1, int y1, int x2, int y2, int thick);
void Draw3LineWInv(int x1, int y1, int x2, int y2, int thick);

// ----- Draw round (filled circle)

//  d ... diameter of the round (radius = d/2)
//  mask ... hide parts of the round with DRAWCAN_ROUND_* (or their combination); use DRAWCAN_ROUND_ALL or 0 to draw whole round

// Draw round (does not check clipping limits; diameter must be > 0)
void DrawCan3Round_Fast(sDrawCan* can, int x, int y, int d, u16 col, u8 mask);
void Draw3Round_Fast(int x, int y, int d, u16 col, u8 mask);

// Draw round (checks clipping limits)
void DrawCan3Round(sDrawCan* can, int x, int y, int d, u16 col, u8 mask);
void Draw3Round(int x, int y, int d, u16 col, u8 mask);

// Draw round inverted (does not check clipping limits; diameter must be > 0)
void DrawCan3RoundInv_Fast(sDrawCan* can, int x, int y, int d, u8 mask);
void Draw3RoundInv_Fast(int x, int y, int d, u8 mask);

// Draw round inverted (checks clipping limits)
void DrawCan3RoundInv(sDrawCan* can, int x, int y, int d, u8 mask);
void Draw3RoundInv(int x, int y, int d, u8 mask);

// ----- Draw circle
//  d ... diameter of the circle (radius = d/2)
//  mask ... draw circle arcs, use combination of DRAWCAN_CIRCLE_*; use DRAWCAN_CIRCLE_ALL or 0xff to draw whole circle

// Draw circle or arc (does not check clipping limits)
void DrawCan3Circle_Fast(sDrawCan* can, int x, int y, int d, u16 col, u8 mask);
void Draw3Circle_Fast(int x, int y, int d, u16 col, u8 mask);

// Draw circle or arc (checks clipping limits)
void DrawCan3Circle(sDrawCan* can, int x, int y, int d, u16 col, u8 mask);
void Draw3Circle(int x, int y, int d, u16 col, u8 mask);

// Draw circle or arc inverted (does not check clipping limits)
void DrawCan3CircleInv_Fast(sDrawCan* can, int x, int y, int d, u8 mask);
void Draw3CircleInv_Fast(int x, int y, int d, u8 mask);

// Draw circle or arc inverted (checks clipping limits)
void DrawCan3CircleInv(sDrawCan* can, int x, int y, int d, u8 mask);
void Draw3CircleInv(int x, int y, int d, u8 mask);

// ----- Draw ring
//  d ... outer diameter of the ring (outer radius = d/2)
//  din ... inner diameter of the ring (inner radius = din/2)
//  mask ... hide parts of the ring with DRAWCAN_ROUND_* (or their combination); use DRAWCAN_ROUND_ALL or 0 to draw whole ring

// Draw ring (does not check clipping limits)
void DrawCan3Ring_Fast(sDrawCan* can, int x, int y, int d, int din, u16 col, u8 mask);
void Draw3Ring_Fast(int x, int y, int d, int din, u16 col, u8 mask);

// Draw ring (checks clipping limits)
void DrawCan3Ring(sDrawCan* can, int x, int y, int d, int din, u16 col, u8 mask);
void Draw3Ring(int x, int y, int d, int din, u16 col, u8 mask);

// Draw ring inverted (does not check clipping limits)
void DrawCan3RingInv_Fast(sDrawCan* can, int x, int y, int d, int din, u8 mask);
void Draw3RingInv_Fast(int x, int y, int d, int din, u8 mask);

// Draw ring inverted (checks clipping limits)
void DrawCan3RingInv(sDrawCan* can, int x, int y, int d, int din, u8 mask);
void Draw3RingInv(int x, int y, int d, int din, u8 mask);

// ----- Draw triangle

// Draw triangle (does not check clipping limits)
void DrawCan3Triangle_Fast(sDrawCan* can, int x1, int y1, int x2, int y2, int x3, int y3, u16 col);
void Draw3Triangle_Fast(int x1, int y1, int x2, int y2, int x3, int y3, u16 col);

// Draw triangle (checks clipping limits)
void DrawCan3Triangle(sDrawCan* can, int x1, int y1, int x2, int y2, int x3, int y3, u16 col);
void Draw3Triangle(int x1, int y1, int x2, int y2, int x3, int y3, u16 col);

// Draw triangle inverted (does not check clipping limits)
void DrawCan3TriangleInv_Fast(sDrawCan* can, int x1, int y1, int x2, int y2, int x3, int y3);
void Draw3TriangleInv_Fast(int x1, int y1, int x2, int y2, int x3, int y3);

// Draw triangle inverted (checks clipping limits)
void DrawCan3TriangleInv(sDrawCan* can, int x1, int y1, int x2, int y2, int x3, int y3);
void Draw3TriangleInv(int x1, int y1, int x2, int y2, int x3, int y3);

// ----- Draw fill area

// Draw fill area
void DrawCan3Fill(sDrawCan* can, int x, int y, u16 col);
void Draw3Fill(int x, int y, u16 col);

// ----- Draw character
//  scalex ... scale size in X direction (> 0)
//  scaley ... scale size in Y direction (> 0)

// select font
//  font ... pointer to current font (256 characters in cells of width 8 pixels, 1-bit format)
//  fontw ... font width (number of pixels, max. 8)
//  fonth ... font height (number of lines)
void Draw3SelFont(const u8* font, u8 fontw, u8 fonth);

void Draw3SelFont8x8();		// sans-serif bold, height 8 lines
void Draw3SelFont8x14();	// sans-serif bold, height 14 lines
void Draw3SelFont8x16();	// sans-serif bold, height 16 lines
void Draw3SelFont8x14Serif();	// serif bold, height 14 lines
void Draw3SelFont8x16Serif();	// serif bold, height 16 lines
void Draw3SelFont6x8();		// condensed font, width 6 pixels, height 8 lines
void Draw3SelFont8x8Game();	// game font, height 8 lines
void Draw3SelFont8x8Ibm();	// IBM charset font, height 8 lines
void Draw3SelFont8x14Ibm();	// IBM charset font, height 14 lines
void Draw3SelFont8x16Ibm();	// IBM charset font, height 16 lines
void Draw3SelFont8x8IbmThin();	// IBM charset thin font, height 8 lines
void Draw3SelFont8x8Italic();	// italic, height 8 lines
void Draw3SelFont8x8Thin();	// thin font, height 8 lines
void Draw3SelFont5x8();		// tiny font, width 5 pixels, height 8 lines

// Draw character with transparent background (does not check clipping limits)
void DrawCan3Char_Fast(sDrawCan* can, char ch, int x, int y, u16 col, int scalex, int scaley);
void Draw3Char_Fast(char ch, int x, int y, u16 col, int scalex, int scaley);

// Draw character with transparent background (checks clipping limits)
void DrawCan3Char(sDrawCan* can, char ch, int x, int y, u16 col, int scalex, int scaley);
void Draw3Char(char ch, int x, int y, u16 col, int scalex, int scaley);

// Draw character with background (does not check clipping limits)
void DrawCan3CharBg_Fast(sDrawCan* can, char ch, int x, int y, u16 col, u16 bgcol, int scalex, int scaley);
void Draw3CharBg_Fast(char ch, int x, int y, u16 col, u16 bgcol, int scalex, int scaley);

// Draw character with background (checks clipping limits)
void DrawCan3CharBg(sDrawCan* can, char ch, int x, int y, u16 col, u16 bgcol, int scalex, int scaley);
void Draw3CharBg(char ch, int x, int y, u16 col, u16 bgcol, int scalex, int scaley);

// Draw character inverted (does not check clipping limits)
void DrawCan3CharInv_Fast(sDrawCan* can, char ch, int x, int y, int scalex, int scaley);
void Draw3CharInv_Fast(char ch, int x, int y, int scalex, int scaley);

// Draw character inverted (checks clipping limits)
void DrawCan3CharInv(sDrawCan* can, char ch, int x, int y, int scalex, int scaley);
void Draw3CharInv(char ch, int x, int y, int scalex, int scaley);

// ----- Draw text
//  len ... text length, or -1 to detect ASCIIZ length
//  scalex ... scale size in X direction (> 0)
//  scaley ... scale size in Y direction (> 0)

// Draw text with transparent background (does not check clipping limits)
void DrawCan3Text_Fast(sDrawCan* can, const char* text, int len, int x, int y, u16 col, int scalex, int scaley);
void Draw3Text_Fast(const char* text, int len, int x, int y, u16 col, int scalex, int scaley);

// Draw text with transparent background (checks clipping limits)
void DrawCan3Text(sDrawCan* can, const char* text, int len, int x, int y, u16 col, int scalex, int scaley);
void Draw3Text(const char* text, int len, int x, int y, u16 col, int scalex, int scaley);

// Draw text with background (does not check clipping limits)
void DrawCan3TextBg_Fast(sDrawCan* can, const char* text, int len, int x, int y, u16 col, u16 bgcol, int scalex, int scaley);
void Draw3TextBg_Fast(const char* text, int len, int x, int y, u16 col, u16 bgcol, int scalex, int scaley);

// Draw text with background (checks clipping limits)
void DrawCan3TextBg(sDrawCan* can, const char* text, int len, int x, int y, u16 col, u16 bgcol, int scalex, int scaley);
void Draw3TextBg(const char* text, int len, int x, int y, u16 col, u16 bgcol, int scalex, int scaley);

// Draw text inverted (does not check clipping limits)
void DrawCan3TextInv_Fast(sDrawCan* can, const char* text, int len, int x, int y, int scalex, int scaley);
void Draw3TextInv_Fast(const char* text, int len, int x, int y, int scalex, int scaley);

// Draw text inverted (checks clipping limits)
void DrawCan3TextInv(sDrawCan* can, const char* text, int len, int x, int y, int scalex, int scaley);
void Draw3TextInv(const char* text, int len, int x, int y, int scalex, int scaley);

// ----- Draw ellipse
//  dx ... diameter of ellipse in X direction, range 1..430 (X radius = dx/2)
//  dy ... diameter of ellipse in Y direction, range 1..430 (Y radius = dy/2)
//  mask ... draw ellipse arcs, use combination of DRAWCAN_ELLIPSE_*; use DRAWCAN_ELLIPSE_ALL or 0x0f to draw whole ellipse

// Draw ellipse (does not check clipping limits)
void DrawCan3Ellipse_Fast(sDrawCan* can, int x, int y, int dx, int dy, u16 col, u8 mask);
void Draw3Ellipse_Fast(int x, int y, int dx, int dy, u16 col, u8 mask);

// Draw ellipse (checks clipping limits)
void DrawCan3Ellipse(sDrawCan* can, int x, int y, int dx, int dy, u16 col, u8 mask);
void Draw3Ellipse(int x, int y, int dx, int dy, u16 col, u8 mask);

// Draw ellipse inverted (does not check clipping limits)
void DrawCan3EllipseInv_Fast(sDrawCan* can, int x, int y, int dx, int dy, u8 mask);
void Draw3EllipseInv_Fast(int x, int y, int dx, int dy, u8 mask);

// Draw ellipse inverted (checks clipping limits)
void DrawCan3EllipseInv(sDrawCan* can, int x, int y, int dx, int dy, u8 mask);
void Draw3EllipseInv(int x, int y, int dx, int dy, u8 mask);

// ----- Draw filled ellipse
//  dx ... diameter of ellipse in X direction, range 1..430 (X radius = dx/2)
//  dy ... diameter of ellipse in Y direction, range 1..430 (Y radius = dy/2)
//  mask ... hide parts of the ellipse with DRAWCAN_ROUND_* (or their combination); use DRAWCAN_ROUND_ALL or 0 to draw whole ellipse

// Draw filled ellipse (does not check clipping limits)
void DrawCan38FillEllipse_Fast(sDrawCan* can, int x, int y, int dx, int dy, u16 col, u8 mask);
void Draw3FillEllipse_Fast(int x, int y, int dx, int dy, u16 col, u8 mask);

// Draw filled ellipse (checks clipping limits)
void DrawCan3FillEllipse(sDrawCan* can, int x, int y, int dx, int dy, u16 col, u8 mask);
void Draw3FillEllipse(int x, int y, int dx, int dy, u16 col, u8 mask);

// Draw filled ellipse inverted (does not check clipping limits)
void DrawCan3FillEllipseInv_Fast(sDrawCan* can, int x, int y, int dx, int dy, u8 mask);
void Draw3FillEllipseInv_Fast(int x, int y, int dx, int dy, u8 mask);

// Draw filled ellipse inverte (checks clipping limits)
void DrawCan3FillEllipseInv(sDrawCan* can, int x, int y, int dx, int dy, u8 mask);
void Draw3FillEllipseInv(int x, int y, int dx, int dy, u8 mask);

// ----- Draw image
//  can ... destination canvas
//  xd ... destination X coordinate
//  yd ... destination Y coordinate
//  src ... source image
//  xs ... source X coordinate
//  ys ... source Y coordinate
//  w ... source width to draw
//  h ... source height to draw
//  wbs ... pitch of source image (length of line in bytes)
//  col ... key transparent color

// Draw image with the same format as destination
void DrawCan3Img(sDrawCan* can, int xd, int yd, const void* src, int xs, int ys, int w, int h, int wbs);
void Draw3Img(int xd, int yd, const void* src, int xs, int ys, int w, int h, int wbs);

// Draw image inverted with the same format as destination
void DrawCan3ImgInv(sDrawCan* can, int xd, int yd, const void* src, int xs, int ys, int w, int h, int wbs);
void Draw3ImgInv(int xd, int yd, const void* src, int xs, int ys, int w, int h, int wbs);

// Draw transparent image with the same format as destination
void DrawCan3Blit(sDrawCan* can, int xd, int yd, const void* src, int xs, int ys, int w, int h, int wbs, u16 col);
void Draw3Blit(int xd, int yd, const void* src, int xs, int ys, int w, int h, int wbs, u16 col);

// Draw transparent image inverted with the same format as destination
void DrawCan3BlitInv(sDrawCan* can, int xd, int yd, const void* src, int xs, int ys, int w, int h, int wbs, u16 col);
void Draw3BlitInv(int xd, int yd, const void* src, int xs, int ys, int w, int h, int wbs, u16 col);

// Draw transparent image with the same format as destination, with substitute color
//  fnd ... color to find for
//  subs ... color to substitute with
void DrawCan3BlitSubs(sDrawCan* can, int xd, int yd, const void* src, int xs, int ys, int w, int h, int wbs, u16 col, u16 fnd, u16 subs);
void Draw3BlitSubs(int xd, int yd, const void* src, int xs, int ys, int w, int h, int wbs, u16 col, u16 fnd, u16 subs);

// Get image from canvas to buffer
//  can ... source canvas
//  xs ... source X coordinate
//  ys ... source Y coordinate
//  w ... width
//  h ... height
//  dst ... destination buffer
//  xd ... destination X coordinate
//  yd ... destination Y coordinate
//  wbd ... pitch of destination buffer (length of line in bytes)
void DrawCan3GetImg(const sDrawCan* can, int xs, int ys, int w, int h, void* dst, int xd, int yd, int wbd);
void Draw3GetImg(int xs, int ys, int w, int h, void* dst, int xd, int yd, int wbd);

#if USE_DRAWCAN0		// 1=use DrawCan common functions, if use drawing canvas
// drawing function interface
extern const sDrawCanFnc DrawCan3Fnc;
#endif

#ifdef __cplusplus
}
#endif

#endif // _LIB_DRAWCAN3_H

#endif // USE_DRAWCAN
